/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.igraphique.swing;

import java.awt.Color ;
import java.awt.Dimension ;
import java.awt.Graphics ;
import java.awt.image.BufferedImage ;

import javax.swing.JComponent ;



/**
 * Composant affichant une image sur un fond de couleur donnée.<br>
 * Le composant prend la taille de l'image à afficher ou une taille fixe si on lui en a donné une.
 */
public class JImage extends JComponent
{

    private Color         couleur ;             // Couleur à peindre sur le fond du panneau, s'il n'y a pas d'image
    private BufferedImage image ;               // Image à afficher sur le panneau
    private Dimension     tailleComposant ;     // Taile à donnre au composant
    
    
    
    /**
     * Constructeur.<br>
     * L'image et la taille du composant sont optionnelles.
     * @param couleur         La couleur de fond du composant.
     * @param image           Image à peindre sur le composant.
     * @param tailleComposant Taille à donner au composant.
     */
    public JImage (Color couleur, BufferedImage image, Dimension tailleComposant)
    {
        // Vérifier les paramètres
        if (couleur == null)
            throw new IllegalArgumentException ("La couleur doit être renseignée") ;
        
        // Stocker les valeurs des paramètres
        this.image           = image ;
        this.couleur         = couleur ;
        this.tailleComposant = tailleComposant ;
        
        // Déterminer la taille du composant
        ajusterTaille() ;
    }
    
    
    // Ajuste la taille du composant
    private void ajusterTaille ()
    {
        Dimension taille ;
        
        // Calculer la taille à donner au composant
        if (this.tailleComposant != null)
            taille = this.tailleComposant ;
        else if (this.image != null)
            taille = new Dimension (this.image.getWidth(), this.image.getHeight()) ;
        else
            taille = new Dimension (0, 0) ;
        
        // Indiquer au composant de prendre cette taille autant que possible
        this.setPreferredSize (taille) ;
        this.setMinimumSize   (taille) ;
        this.setMaximumSize   (taille) ;
    }
    
    
    /**
     * Modifie l'image affichée par le composant.
     * @param couleur Nouvelle couleur.
     */
    public void modifCouleur (Color couleur)
    {
        if (couleur == null)
            throw new IllegalArgumentException ("La couleur doit être renseignée") ;
        this.couleur = couleur ;
    }
    
    /**
     * Modifie l'image affichée par le composant.
     * @param image La nouvelle image.
     */
    public void modifImage (BufferedImage image)
    {
        this.image = image ;
        ajusterTaille() ;
    }
    
    /**
     * Modifie la taille à donner au composant.
     * @param taille La nouvelle taille.
     */
    public void modifTaille (Dimension taille)
    {
        this.tailleComposant = taille ;
        ajusterTaille() ;
    }

    
    
    // Peint le composant
    protected void paintComponent (Graphics g)
    {
        // Créer une image pour préparer le dessin
        BufferedImage img = new BufferedImage (this.getWidth(), this.getHeight(), BufferedImage.TYPE_INT_ARGB) ;
        Graphics gImg = img.getGraphics() ;
        
        // Peindre le fond du composant
        gImg.setColor (this.couleur) ;
        gImg.fillRect (0, 0, this.getWidth(), this.getHeight()) ;
        
        // Peindre l'image
        if (this.image != null)
            gImg.drawImage (this.image, 0, 0, this) ;
        
        // Peindre l'image sur le composant
        g.drawImage (img, 0, 0, this) ;
    }
    
    
}
