/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.stockage;

import static fr.histoiremondiale.histoire.utiles.es.fichiers.Fichiers.chargerContenuTexte;
import static fr.histoiremondiale.histoire.utiles.es.fichiers.Proprietes.chargerProprietes;
import static fr.histoiremondiale.histoire.utiles.es.fichiers.Proprietes.sauverProprietes;
//import static org.h2.server.web.PageParser.escapeHtml;
import static org.apache.commons.lang.StringEscapeUtils.escapeHtml ;

import java.io.File ;
import java.io.IOException ;
import java.sql.ResultSet ;
import java.sql.SQLException ;
import java.text.DecimalFormat;
import java.util.ArrayList ;
import java.util.Collections ;
import java.util.HashMap ;
import java.util.List ;
import java.util.Map ;
import java.util.Properties ;
import java.util.Map.Entry ;

import fr.histoiremondiale.histoire.utiles.math.PointSphere;
import fr.histoiremondiale.histoire.DonneesHistoriques;
import fr.histoiremondiale.histoire.DonneesIGraphique;
import fr.histoiremondiale.histoire.donnees.Territoire;
import fr.histoiremondiale.histoire.igraphique.donnees.InfosParagrapheNavig;



/**
 * Classe d'accès aux données pour l'interface graphique.
 */
public class AccesDonneesIGraphique extends AccesDonneesBase
{
    
    public static final String ENCODAGE_PAGES_HTML         = "UTF-8" ;                                      // Encodage des pages html
    public static final String NOM_REP_DONNEES_PARAGRAPHES = "histoire" + File.separator + "paragraphes" ;
    public static final String PREFIXE_NOM_FIC_PARAGRAPHES = "paragraphe" ;                                 // Préfixe des fichiers des paragraphes
    public static final String SUFFIXE_NOM_FIC_PARAGRAPHES = ".txt" ;                                       // Suffixe des fichiers des paragraphes
    public static final String ENCODAGE_FIC_POSITIONS      = "UTF-8" ;
    public static final String NOM_FIC_DEBUT_HTML          = "debutPage.txt" ;                              // Nom du fichier contenant le début des pages html
    public static final String NOM_FIC_FIN_HTML            = "finPage.txt" ;                                // Nom du fichier contenant la fin   des pages html
    public static final String CHEM_FIC_POSITIONS          = System.getProperty ("user.home") + File.separator + ".config/histoiremondiale/positions.conf".replace ('/', File.separatorChar) ;
    public static final String ENCODAGE_FIC_LOUPES         = "UTF-8" ;
    public static final String CHEM_FIC_LOUPES             = System.getProperty ("user.home") + File.separator + ".config/histoiremondiale/loupes.conf".replace ('/', File.separatorChar) ;
    
    private String chemRepDonnees ;
    private String chemRepIcones ;
    

    
    /**
     * Constructeur.
     * @param chemRepDonnees Chemin du répertoire des données de l'application.
     */
    public AccesDonneesIGraphique (String chemRepDonnees, String chemRepIcones)
    {
        super (chemRepDonnees) ;
        this.chemRepDonnees = chemRepDonnees ;
        this.chemRepIcones  = chemRepIcones;
    }
    
    
    
    /**
     * Sauvegarde les informations nécessaires à l'interface graphique
     *   (du moins celles qui doivent être sauvegardées).
     * @param donneesIGraphique Données de l'interface graphique.
     * @throws IOException En cas de problème d'accès aux fichiers.
     */
    public void sauverInfosIGraphique (DonneesIGraphique donneesIGraphique) throws IOException
    {
        sauverPositionTextes         (donneesIGraphique) ;
        sauverValeursLoupeTerritoire (donneesIGraphique) ;
    }
    

    /**
     * Charge la liste des paragraphes affichables dans le navigateur web.<br>
     * Les paragraphes sont triés dans leur ordre d'affichage.
     * @param donneesHisto Données historiques.
     * @return La liste des paragraphes.
     * @throws IOException En cas de problème d'accès aux fichiers.
     */
    public List<InfosParagrapheNavig> chargerInfosParagraphesNavig (DonneesHistoriques donneesHisto) throws IOException
    {
//    	System.out.println("\t\t\t\tChargement des paragraphes");
    	long dateDeb = System.currentTimeMillis() ;
    	long debut = dateDeb;
        try
        {
            String txtRequeteParagraphes      = "SELECT *" +
            		                            "  FROM paragraphes" ;
            
            String txtRequeteLiensParagraphes = "SELECT *" +
                                                "  FROM suivants_paragraphes";
            
            // Charger les paragraphes
            Map<Integer,InfosParagrapheNavig> paragraphesParId = new HashMap<Integer, InfosParagrapheNavig>() ;
            ResultSet                         resParagraphes   = connexionCourante().prepareStatement(txtRequeteParagraphes).executeQuery() ;
            while (resParagraphes.next())
            {
                InfosParagrapheNavig paragraphe = new InfosParagrapheNavig (resParagraphes.getInt    ("id"),
                                                                            donneesHisto.civilisationDId (resParagraphes.getInt ("id_civilisation")),
                                                                            resParagraphes.getInt    ("debut"),
                                                                            resParagraphes.getInt    ("fin"),
                                                                            resParagraphes.getString ("titre")) ;
                paragraphesParId.put (paragraphe.id(), paragraphe) ;
            }
            long dateFin = System.currentTimeMillis() ;
            System.out.println("\t\t\t\tfin du chargement des paragraphes en : " + (dateFin - dateDeb) + " ms");
//            System.out.println("début du chargement des liens entre paragraphes");
            dateDeb = System.currentTimeMillis() ;
            // Charger les liens entre les paragraphes
            ResultSet resLiens = connexionCourante().prepareStatement(txtRequeteLiensParagraphes).executeQuery() ;
            while (resLiens.next())
            {
                InfosParagrapheNavig parCourant = paragraphesParId.get (resLiens.getInt ("id_courant")) ;
                InfosParagrapheNavig parSuivant = paragraphesParId.get (resLiens.getInt ("id_suivant")) ;
                parCourant.suivants().add   (parSuivant) ;
                parSuivant.precedents().add (parCourant) ;
            }
            
            // Fermer les objets de résultat
            resParagraphes.close() ;
            resLiens.close() ;
            dateFin = System.currentTimeMillis() ;
//            System.out.println("\t\t\t\tfin du chargement des liens entre paragraphes en : " + (dateFin - dateDeb) + " ms");
//            System.out.println("début du chargement du texte des paragraphes");
            dateDeb = System.currentTimeMillis() ;
            // Charger le texte html des paragraphes
            // (note performances : c'est le chargement des données qui est long dans cette méthode.
            //  Peut-être qu'avec moins de fichiers éparpillés ce serait plus rapide, mais ça pose d'autres
            //  problèmes (données moins évidentes) ; l'utilisation de StringBuilder plutôt que la
            //  concaténation des chaînes n'a pas vraiment accéléré le processus ; si ça pose un problème,
            //  on pourrait tenter de faire ça en arrière-plan, comme l'affichage des paragraphes)
            // (début et fin de page)
            String chemRepParagraphes = this.chemRepDonnees + File.separator + NOM_REP_DONNEES_PARAGRAPHES + File.separator ;
            String debPage            = chargerContenuTexte (chemRepParagraphes + NOM_FIC_DEBUT_HTML, ENCODAGE_PAGES_HTML) ;
            String finPage            = chargerContenuTexte (chemRepParagraphes + NOM_FIC_FIN_HTML,   ENCODAGE_PAGES_HTML) ;
            String bibliographie      = "<div align=\"center\">"
                                      + "<a href=\"bibliographie.html\">" 
                                      + "<img src = \"file:" + this.chemRepIcones + File.separator + "booksmini16.png\""
                                      + "     alt = \"Bibliographie\" border = \"0\"></a></div>" ;
            // (paragraphes)
            for (InfosParagrapheNavig paragraphe : paragraphesParId.values())
            {
            	String s = chemRepParagraphes
                        + PREFIXE_NOM_FIC_PARAGRAPHES
                        + new DecimalFormat("0000").format (paragraphe.id())
                        + SUFFIXE_NOM_FIC_PARAGRAPHES;
            	File f = new File(s);
            	String corpsParagraphe = "<h1>Paragraphe absent</h1>";
            	if (f.exists()) 
            	{
	                // Créer ou charger les morceaux
	                // (corps du paragraphe)
	                corpsParagraphe = new StringBuilder()
	                                                 .append ("<br>")
	                                                 .append (chargerContenuTexte (chemRepParagraphes
	                                                                                 + PREFIXE_NOM_FIC_PARAGRAPHES
	                                                                                 + new DecimalFormat("0000").format (paragraphe.id())
	                                                                                 + SUFFIXE_NOM_FIC_PARAGRAPHES,
	                                                                               ENCODAGE_PAGES_HTML))
	                                                 .append ("<br>")
	                                                 .toString() ;
            	}
            	// (titre du paragraphe)
            	StringBuilder titreTmp = new StringBuilder() ;
            	titreTmp.append ("\n<strong style=\"color:red\">") ;
            	if (paragraphe.civilisation() != null)
            	{
            		titreTmp.append (escapeHtml (paragraphe.civilisation().nom())) ;
	                titreTmp.append (" (") ;
	                titreTmp.append (paragraphe.anneeDeb()) ;
	              	titreTmp.append (" à ") ;
	             	titreTmp.append (paragraphe.anneeFin()) ;
	             	titreTmp.append (") : ") ;
            	}
	            titreTmp.append (escapeHtml (paragraphe.titre())) ;
	            titreTmp.append ("</strong>\n\n") ;
	         	String titre = titreTmp.toString() ;
	         	// (liens)
	         	StringBuilder liensPrecSuiv = new StringBuilder() ;
                if (paragraphe.precedents().size() + paragraphe.suivants().size() > 0)
                {
                    // liensPrecSuiv.append ("\n<div align=right>\n") ;
                    liensPrecSuiv.append ("<table width=\"100%\">") ;
                    liensPrecSuiv.append ("<tr>") ;
                    // (précédents)
                    liensPrecSuiv.append ("<td>") ;
                    int cptPrecedents = 0 ;
                    for (InfosParagrapheNavig parPrecedent : paragraphe.precedentsTries())
                    {
                        String txtLien = "Précédent" +
                                         (parPrecedent.civilisation() != null && ! parPrecedent.civilisation().equals (paragraphe.civilisation()) ?
                                             " : " + escapeHtml (parPrecedent.civilisation().nom()) :
                                             "") ;
                        if (cptPrecedents++ > 0)
                            liensPrecSuiv.append ("<br>") ;
                        liensPrecSuiv.append ("<a href=\"#" + escapeHtml (parPrecedent.ancre()) + "\">" + txtLien + "</a>\n") ;
                    }
                    liensPrecSuiv.append ("</td>") ;
                    // (suivants)
                    liensPrecSuiv.append ("<td align=\"right\">") ;
                    int cptSuivants = 0 ;
                    for (InfosParagrapheNavig parSuivant : paragraphe.suivantsTries())
                    {
                        String txtLien = "Suivant" +
                                         (parSuivant.civilisation() != null && ! parSuivant.civilisation().equals (paragraphe.civilisation()) ?
                                             " : " + escapeHtml (parSuivant.civilisation().nom()) :
                                             "") ;
                        if (cptSuivants++ > 0)
                            liensPrecSuiv.append ("<br>") ;
                        liensPrecSuiv.append ("<a href=\"#" + escapeHtml (parSuivant.ancre()) + "\">" + txtLien + "</a>\n") ;
                    }
                    liensPrecSuiv.append ("</td>") ;
                    liensPrecSuiv.append ("</tr>") ;
                    liensPrecSuiv.append ("</table>") ;
                    // liensPrecSuiv.append ("</div>\n\n") ;
                }
                
				// Assembler la page
                paragraphe.modifCodeHtml (new StringBuilder()
                                                  .append (debPage)
                                                  .append ("<br>")
                                                  .append (titre)
                                                  .append (liensPrecSuiv)
                                                  .append (corpsParagraphe)
                                                  .append (liensPrecSuiv)
                                                  .append (bibliographie)
                                                  .append ("<br>")
                                                  .append (finPage)
                                                  .toString()) ;
                paragraphe.modifCodeHtmlSimple (new StringBuilder()
                                                        .append (titre)
                                                        .append (corpsParagraphe)
                                                        .toString()) ;
            }
            dateFin = System.currentTimeMillis() ;
            System.out.println("\t\t\t\tfin du chargement du texte des paragraphes en : " + (dateFin - dateDeb) + " ms");

//            System.out.println("début du tri des paragraphes");
            dateDeb = System.currentTimeMillis() ;
            // Renvoyer les paragraphes
            List<InfosParagrapheNavig> paragraphes = new ArrayList<> (paragraphesParId.values()) ;
            Collections.sort (paragraphes) ;
            dateFin = System.currentTimeMillis() ;
            System.out.println("\t\t\t\tfin du tri des paragraphes en : " + (dateFin - dateDeb) + " ms");
            
            System.out.println("\t\t\t\ttemps total pour les paragraphes : " + (dateFin - debut) + " ms");
            return paragraphes ;
        }
        catch (SQLException e)
        {
            throw new RuntimeException ("Erreur lors du chargement des données", e) ;
        }
    }
    
     /**
     * Charge la liste des positions des textes dont la position a été modifiée par l'utilisateur.
     * @param donneesHisto Données historiques.
     * @return La table d'association des territoires et de la position de leur texte (si déplacé).
     * @throws IOException En cas de problème d'accès aux fichiers.
     */
    public Map<Territoire,PointSphere> chargerPositionTextes (DonneesHistoriques donneesHisto) throws IOException
    {
        // Charger les données de la sauvegarde
        Properties props = chargerProprietes (CHEM_FIC_POSITIONS, ENCODAGE_FIC_POSITIONS) ;
        
        // Extraire les informations
        Map<Territoire,PointSphere> positionsTextes = new HashMap<Territoire, PointSphere>() ;
        for (Entry<Object,Object> position : props.entrySet())
        {
            int    idTerritoire = Integer.parseInt   (""+position.getKey()) ;
            double longitude    = Double.parseDouble (position.getValue().toString().split(",")[0]) ;
            double latitude     = Double.parseDouble (position.getValue().toString().split(",")[1]) ;
            Territoire territoire = donneesHisto.territoireParId (idTerritoire) ;
            if (territoire != null)
                positionsTextes.put (territoire, new PointSphere (longitude, latitude)) ;
            // (compatibilité descendante : conserver les données de territoires inconnus dans cette version)
            else
                positionsTextes.put (new Territoire (idTerritoire), new PointSphere (longitude, latitude)) ;
        }
        
        // Renvoyer les informations chargées
        return positionsTextes ;
    }
    
    
    /**
     * Sauvegarde la liste des positions des textes dont la position a été modifiée par l'utilisateur.
     * @param donneesIGraphique Données de l'interface graphique.
     * @throws IOException En cas de problème d'accès aux fichiers.
     */
    public void sauverPositionTextes (DonneesIGraphique donneesIGraphique) throws IOException
    {
        // Stocker les valeurs sous forme de propriétés
        Properties props = new Properties() ;
        for (Entry<Territoire,PointSphere> positionTexte : donneesIGraphique.centresTextesTerritoires().entrySet())
        {
            Territoire  territoire = positionTexte.getKey() ;
            PointSphere ptCentral  = positionTexte.getValue() ;
            // (test du territoire null ajouté par Patrice : à cause d'un ancien bug ? mieux vaut éviter de planter l'appli pour ça...)
            if (territoire !=null) props.put ("" + territoire.id(), "" + ptCentral.longitude() + "," + ptCentral.latitude()) ;
        }
        
        // Sauvegarder les valeurs
        sauverProprietes (props, "Position des textes déplacés de HistoireMondiale", CHEM_FIC_POSITIONS, ENCODAGE_FIC_POSITIONS) ;
    }

    
    /**
     * Charge la liste des niveaux de loupe de territoire modifiés par l'utilisateur.
     * @param donneesHisto Données historiques.
     * @return La table d'association des territoires et de leur valeur limite de loupe.
     * @throws IOException En cas de problème d'accès aux fichiers.
     */
    public Map<Territoire,Double> chargerValeursLoupeTerritoire (DonneesHistoriques donneesHisto) throws IOException
    {
        // Charger les données de la sauvegarde
        Properties props = chargerProprietes (CHEM_FIC_LOUPES, ENCODAGE_FIC_LOUPES) ;
        
        // Extraire les informations
        Map<Territoire,Double> valsLoupesTextesTerritoires = new HashMap<Territoire, Double>() ;
        for (Entry<Object,Object> valLoupeTerritoire : props.entrySet())
        {
            int    idTerritoire = Integer.parseInt   (""+valLoupeTerritoire.getKey()) ;
            double valLoupe     = Double.parseDouble (""+valLoupeTerritoire.getValue()) ;
            Territoire territoire = donneesHisto.territoireParId (idTerritoire) ;
            if (territoire != null)
                valsLoupesTextesTerritoires.put (territoire, valLoupe) ;
            // (compatibilité descendante : conserver les données sur des territoires inconnus dans cette version)
            else
                valsLoupesTextesTerritoires.put (new Territoire (idTerritoire), valLoupe) ;
        }
        
        // Renvoyer les informations chargées
        return valsLoupesTextesTerritoires ;
    }
    
    
    /**
     * Sauvegarde la liste des niveaux de loupe de territoire modifiés par l'utilisateur.
     * @param donneesIGraphique Données de l'interface graphique.
     * @throws IOException En cas de problème d'accès aux fichiers.
     */
    public void sauverValeursLoupeTerritoire (DonneesIGraphique donneesIGraphique) throws IOException
    {
        // Stocker les valeurs sous forme de propriétés
        Properties props = new Properties() ;
        for (Entry<Territoire,Double> valLoupeTexteTerritoire : donneesIGraphique.loupesTextesTerritoires().entrySet())
        {
            Territoire territoire = valLoupeTexteTerritoire.getKey() ;
            Double     valLoupe   = valLoupeTexteTerritoire.getValue() ;
            // (test du territoire null ajouté par Patrice : à cause d'un ancien bug ? mieux vaut éviter de planter l'appli pour ça...)
            if (territoire != null) props.put ("" + territoire.id(), "" + valLoupe) ;
        }
        
        // Sauvegarder les valeurs
        sauverProprietes (props,
                          "Valeurs de loupe minimale modifiées pour les textes des territoires de Histoire Mondiale",
                          CHEM_FIC_LOUPES,
                          ENCODAGE_FIC_LOUPES) ;
    }
    
    
}
