package fr.histoiremondiale.histoire.igraphique.composants;

import java.awt.Point;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.PlainDocument;

import fr.histoiremondiale.histoire.utiles.igraphique.swing.Messages;
/**
 * Un composant pour ne rentrer que des nombres entiers et qui au retour ne s'affiche pas dans la locale
 * Avec un JFormatedTextField numérique, l'affichage utilise la locale de l'ordinateur pour afficher la
 * saisie, c'est à dire en particulier regrouper les chiffres par groupes de 3 séparés par un espace.
 * On peut préciser si ce composant accepte les nombres relatifs (possibilité d'un signe -), si les 
 * nombres peuvent commencer par 0, les valeurs extrèmes pouvant être entrées.
 * Si on veut limiter les nombre de chiffres, il faut passer par les valeurs extrèmes.
 * Un nombre entre 3 et 5 chiffres sera donc compris entre 100 et 99999 s'il n'est pas relatif et
 * entre -99999 et -100 ou entre 100 et 99999 s'il est relatif.
 * Il y a deux classes : le JTextField lui-même et DocumentNumerique
 * @author Patrice
 *
 */
public class TextFieldNumerique extends JTextField 
{
	boolean relatif = false;
	boolean valeursEncadrees = false;
	List<Point> valeursAutorisees = new ArrayList<Point>() ;
	
	public TextFieldNumerique()
	{
		super();
		this.setDocument(new DocumentNumerique());
		this.setHorizontalAlignment(JTextField.RIGHT);
	}
	
	public TextFieldNumerique(int value) 
	{
		this();
		setText("" + value);
	}

	public TextFieldNumerique(boolean relatif)
	{
		this();
		this.relatif = relatif;
	}

	public TextFieldNumerique(int value, boolean relatif) 
	{
		this();
		this.relatif = relatif;
		setText("" + value);
	}

	public TextFieldNumerique(int value, boolean relatif, int mini, int maxi) 
	{
		this();
		if (mini > maxi)
		{
			throw new RuntimeException("la valeur mini (" + mini + ") doit être inférieure à la valeur maxi (" + maxi + ")");
		}
		valeursAutorisees.add(new Point(mini, maxi));
		valeursEncadrees = true;
		this.relatif = relatif;
		setText("" + value);
	}

	//autorise les nombres négatifs (par defaut false)
	public void setRelatif(boolean b)
	{
		relatif = b;
	}

	public static void main(String[] args) 
	{
		JFrame fenetre = new JFrame();
		fenetre.setSize(600, 100);
		fenetre.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		JPanel panneau = new JPanel();
		
		JLabel label = new JLabel("entrer des chiffres");
		TextFieldNumerique tfn = new TextFieldNumerique(-1800, true);
		tfn.addValeursAutorisees(-3100,-1);
		tfn.addValeursAutorisees(1,2000);
		tfn.setHorizontalAlignment(JTextField.RIGHT);
		tfn.setColumns(4);

		panneau.add(label);
		panneau.add(tfn);

		fenetre.getContentPane().add(panneau);
		fenetre.setVisible(true);
	}
	
	public void addValeursAutorisees(int mini, int maxi) 
	{
		if (mini > maxi)
		{
			throw new RuntimeException("la valeur mini (" + mini + ") doit être inférieure à la valeur maxi (" + maxi + ")");
		}
		valeursAutorisees.add(new Point(mini,maxi));
		valeursEncadrees = true;
	}

	class DocumentNumerique extends PlainDocument
	{
		@Override
		public void insertString(int offset, String str, AttributeSet a) throws BadLocationException 
		{
			//s'il n'y a rien à insérer 
			if (str.isEmpty()) return;
			//la valeur avant changement
			String oldValue = this.getText(0, this.getLength());
			//on veut insérer quelque chose avant le signe -, ce n'est pas autorisé
			if (offset == 0 && oldValue.startsWith("-")) return;
			//la valeur à introduire à l'emplacement offset
			//la cas de la possibilité d'un nombre négatif et donc de l'introduction ou non
			//d'un signe - est traité dans la fonction numerique
			String s = numerique(offset,str);
			//la valeur à insérer  n'est pas valide
			if (s.isEmpty()) return;
			//la nouvelle valeur obtenue
			String newValue = "";
			char[] oldChiffres = oldValue.toCharArray();
			char[] newChiffres = s.toCharArray();
			
			for(int i = 0; i < offset; i++) newValue += oldChiffres[i];
			for (int i = 0; i < newChiffres.length; i++) newValue += newChiffres[i];
			for (int i = offset; i < oldChiffres.length; i++) newValue+= oldChiffres[i];
			boolean valide = (newValue.equals("-"));
			if(!valide)
			{
				int val = Integer.parseInt(newValue);
				if (valeursEncadrees)
				{
					for(Point pt : valeursAutorisees)
					{
						if (pt.x <= val  && val <= pt.y) 
						{
							valide = true;
							break;
						}
					}
					if (!valide)
					{
						Messages.messageInfo("Les valeurs autorisées se trouvent dans la liste d'intervalles " + chaine(valeursAutorisees));
						return;
					}
				}
				else valide = true;
			}
			if (valide) super.insertString(offset, s, a);
		}

		private String chaine(List<Point> valeursAutorisees) 
		{
			String s = "(";
			for(Point pt : valeursAutorisees)
			{
				s += ",[" + pt.x + " à " + pt.y + "]";
			}
			s = s.replace("(,", "(") + ")";
			return s ;
		}
	}

	//seuls les chiffres sont utilisés
	//et éventuellement le signe -
	//la valeur str peut être introduite via un copier-coller
	//donc ne se limite pas à un seul caractère
	public String numerique(int offset, String str) 
	{
		if (str.isEmpty()) return str;
		char[] chiffres = str.toCharArray();
		String s = "";
		
		if (relatif && offset == 0 && chiffres[0] == '-') s="-";
		for(int i = 0; i < chiffres.length; i++)
		{ 
			char c = chiffres[i];
			if (Character.isDigit(c)) s+= c;
		}
		return s;
	}
	
	public void setValue(int value) 
	{
		setText("" + value);
	}
}
