/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire;

import java.io.IOException ;
import java.util.ArrayList ;
import java.util.HashMap ;
import java.util.List ;
import java.util.Map ;

import fr.histoiremondiale.histoire.donnees.Civilisation;
import fr.histoiremondiale.histoire.donnees.Territoire;
import fr.histoiremondiale.histoire.igraphique.donnees.InfosParagrapheNavig;
import fr.histoiremondiale.histoire.stockage.AccesDonneesIGraphique;
import fr.histoiremondiale.histoire.utiles.igraphique.swing.Observable;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;



/**
 * Données nécessaires à l'interface graphique.
 */
// Note : Les informations lues par le traceur de carte doivent être accédées à travers des méthodes
//        synchronisées (le traceur travaille dans son propre fil d'exécution et certaines informations
//        sont modifiables).
public class DonneesIGraphique extends Observable
{

    // Données
    private List<InfosParagrapheNavig>  paragraphes ;                   // Paragraphes du navigateur html
    private Map<Territoire,PointSphere> centresTextesTerritoires ;      // Points centraux des textes des territoires (quand le point central a été déplacé)
    private boolean                     donneesTraceCarteModifiees ;    // Indique si les données nécessaires au tracé de la carte ont été modifiées depuis la dernière fois qu'on a demandé
    private Map<Territoire,Double>      loupesTextesTerritories ;       // Valeurs minimales de loupes pour l'affichage des territoires (si modifiées par rapport aux originales)

    
    // Accès aux données
    private AccesDonneesIGraphique      accesDonneesIGraphique ;
    
    

    /**
     * Constructeur.
     * @param accesDonneesIGraphique Objet d'accès aux données pour l'interface graphique.
     * @param donneesHisto           Données historiques.
     */
    public DonneesIGraphique (AccesDonneesIGraphique accesDonneesIGraphique, DonneesHistoriques donneesHisto)
    {
        long debut = System.currentTimeMillis() ;
        long dateFin, dateDeb;
        System.out.println ("\t\tDébut du chargement des données pour l'interface graphique") ;
        
        try
        {
//        	System.out.println("\t\t\tdébut de l'initialisation de l'accès aux données graphiques");
//        	dateDeb = System.currentTimeMillis() ;
            // Initialiser les objets d'accès aux données
            this.accesDonneesIGraphique = accesDonneesIGraphique ;
//        	dateFin = System.currentTimeMillis() ;
//        	System.out.println("\t\t\tfin de l'initialisation de l'accès aux données graphiques en : " + (dateFin-dateDeb) + " ms");
           
//        	System.out.println("\tdébut du chargement des paragraphes");
        	dateDeb = System.currentTimeMillis() ;
            // Remplir les données qui sont chargées en permanence en mémoire
            // (paragraphes)
            this.paragraphes              = this.accesDonneesIGraphique.chargerInfosParagraphesNavig  (donneesHisto) ;
        	dateFin = System.currentTimeMillis() ;
        	System.out.println("\t\t\tfin du chargement des paragraphes en : " + (dateFin-dateDeb) + " ms");
            
//        	System.out.println("\tdébut du chargement de la position des textes");
//        	dateDeb = System.currentTimeMillis() ;
            // (positions des textes déplacés)
            try
            {
                this.centresTextesTerritoires = this.accesDonneesIGraphique.chargerPositionTextes         (donneesHisto) ;
            }
            catch (Throwable e)
            {
                throw new RuntimeException ("Erreur lors du chargement des positions des textes déplacés.\n"
                		+ "Vous pouvez essayer de détruire le fichier de configuration contenant les positions modifiées (vos modifications seront perdues).", e) ;
            }
//        	dateFin = System.currentTimeMillis() ;
//        	System.out.println("\t\t\tfin du chargement de la position des textes en : " + (dateFin-dateDeb) + " ms");

//        	System.out.println("\tdébut du chargement des limites de visibilité");
//        	dateDeb = System.currentTimeMillis() ;
        	// (niveaux de limite de visibilité modifiés)
            try
            {
                this.loupesTextesTerritories  = this.accesDonneesIGraphique.chargerValeursLoupeTerritoire (donneesHisto) ;
            }
            catch (Throwable e)
            {
                throw new RuntimeException ("Erreur lors du chargement des niveaux de grossissement limite de visibilité des textes.\n"
                		+ "Vous pouvez essayer de détruire le fichier de configuration contenant les niveaux modifiés (vos modifications seront perdues).", e) ;
            }
        	dateFin = System.currentTimeMillis() ;
//        	System.out.println("\t\t\tfin du chargement des limites de visibilité en : " + (dateFin-dateDeb) + " ms");
    
            dateFin = System.currentTimeMillis() ;
            System.out.println ("\t\tfin du chargement des données pour l'interface graphique en : " + (dateFin - debut) + " ms.") ;
//            System.out.println ("\tNombre de paragraphes chargés           : " + this.paragraphes().size()) ;
//            System.out.println ("\tNombre de positions de texte chargées   : " + this.centresTextesTerritoires().size()) ;
//            System.out.println ("\tNombre de loupes de territoire chargées : " + this.loupesTextesTerritoires().size()) ;
        }
        catch (IOException e)
        {
            throw new RuntimeException ("Erreur lors du chargement des données pour l'interface graphique", e) ;
        }
        
    }
    
    
    /**
     * Renvoie le paragraphe contenant l'ancre en paramètre.<br>
     * Si aucun paragraphe ne correspond, renvoie null.
     * @param ancre L'ancre recherchée.
     * @return Le paragraphe correspondant à l'ancre donnée, null si aucun.
     */
    public synchronized InfosParagrapheNavig paragrapheParAncre (String ancre)
    {
        // Vérifier les paramètres
        if (ancre == null)
            throw new IllegalArgumentException ("La civilisation (ancre) devrait être renseignée") ;
        
        // Chercher un paragraphe correspondant aux critères
        for (InfosParagrapheNavig paragraphe : this.paragraphes)
        {
            if (ancre.equals (paragraphe.ancre()))
                return paragraphe ;
        }
        return null ;
    }
    
    
    /**
     * Renvoie le paragraphe correspondant aux critères donnés.<br>
     * (annee : trouver un paragraphe qui couvre cette date)<br>
     * Si aucun paragraphe ne corespond, renvoie null.
     * @param civilisation Critère civilisation.
     * @param annee        Critère année.
     * @return Le paragraphe correspondant à ces critères, null si aucun.
     */
    public synchronized InfosParagrapheNavig paragrapheParCriteres (Civilisation civilisation, Integer annee)
    {
        // Vérifier les paramètres
        if (civilisation == null)
            throw new IllegalArgumentException ("La civilisation devrait être renseignée") ;
        
        // Chercher un paragraphe correspondant aux critères
        for (InfosParagrapheNavig paragraphe : this.paragraphes)
        {
            if (paragraphe.civilisation() != null && paragraphe.civilisation().equals (civilisation) && paragraphe.anneeEstDansPeriodeCouverte (annee))
                return paragraphe ;
        }
        return null ;
    }
    
    
    /**
     * Renvoie les paragraphes correspondant aux critères donnés.<br>
     * Si un critère n'est pas renseigné (null), il n'est pas utilisé pour filtrer les paragraphes.<br>
     * Si aucun paragraphe ne correspond, renvoie une liste vide.
     * @param civilisation Critère civilisation.
     * @param annee        Critère année.
     * @return Les paragraphes correspondant aux critères.
     */
    public synchronized List<InfosParagrapheNavig> paragraphesParCriteres (Civilisation civilisation, Integer annee)
    {
        // Lister les paragraphes correspondant aux critères
        List<InfosParagrapheNavig> paragraphes = new ArrayList<>() ;
        for (InfosParagrapheNavig paragraphe : this.paragraphes)
        {
            if ((civilisation == null || civilisation.equals (paragraphe.civilisation())) &&
                (annee  == null || paragraphe.anneeEstDansPeriodeCouverte (annee)))
            {
                paragraphes.add (paragraphe) ;
            }
        }
        
        // Renvoyer les paragraphes sélectionnés
        return paragraphes ;
    }
    
    
    /**
     * Renvoie le point central des textes d'un territoire.<br>
     * Si ce point a été modifié par l'utilisateur, renvoie la valeur modifiée.<br>
     * Sinon la valeur indiquée dans l'objet Territoire est renvoyée.
     * @param territoire Le territoire considéré.
     * @return Le point central des textes à afficher pour ce territoire.
     */
    public synchronized PointSphere centreTexteTerritoire (Territoire territoire)
    {
        PointSphere centreModifie = this.centresTextesTerritoires.get (territoire) ;
        return (centreModifie != null ? centreModifie : territoire.pointCentral()) ;
    }
    
    
    /**
     * Indique une nouvelle valeur pour le point central du texte d'un territoire.
     * @param territoire Le territoire considéré.
     * @param ptCentral  Le nouveau point central pour les textes de ce territoire.
     */
    public synchronized void modifCentreTexteTerritoire (Territoire territoire, PointSphere ptCentral)
    {
        this.centresTextesTerritoires.put (territoire, ptCentral) ;
        this.donneesTraceCarteModifiees = true ;
        avertirObservateurs ("centreTexteTerritoire") ;
    }

    
    /**
     * Supprime toutes les associations Territoire -> Point central texte.<br>
     * Les points centraux des textes des territoires reviennent à leur valeur par défaut.
     */
    public synchronized void reinitialiserDonneesModifiees ()
    {
        this.centresTextesTerritoires.clear() ;
        this.loupesTextesTerritories.clear ();
        this.donneesTraceCarteModifiees = true ;
        avertirObservateurs ("centreTexteTerritoire") ;
        avertirObservateurs ("loupeTerritoire") ;
    }

    
    /**
     * Renvoie la valeur d'agrandissement limite pour le texte du territoire.<br>
     * Si cette valeur a été modifiée, renvoie la valeur modifiée.<br>
     * Sinon la valeur loupe indiquée dans l'objet Territoire est renvoyée.
     * @param territoire Le territoire considéré.
     * @return La valeur de loupe limite pour afficher les textes de ce territoire.
     */
    public synchronized double loupeTerritoire (Territoire territoire)
    {
        Double loupeModifiee = this.loupesTextesTerritories.get (territoire) ;
        return (loupeModifiee != null ? loupeModifiee : territoire.valLoupeLimiteAff()) ;
    }

    /**
     * Modifie la valeur de loupe associée au territoire.
     * @param territoire Le territoire considéré.
     * @param loupe      La nouvelle valeur de loupe limite.
     */
    public synchronized void modifLoupeTerritoire (Territoire territoire, double loupe)
    {
        this.loupesTextesTerritories.put (territoire, loupe);
        this.donneesTraceCarteModifiees = true ;
        avertirObservateurs ("loupeTerritoire") ;
    }
    
    

    // Accesseurs
    public synchronized List<InfosParagrapheNavig> paragraphes ()
    {
        return this.paragraphes ;
    }
    
    /**
     * Note Une copie est renvoyée : l'objet peut être accédé par différents fils d'exécution.
     * @return Une copie de la table d'association territoire -> centre textes.
     */
    public synchronized Map<Territoire,PointSphere> centresTextesTerritoires ()
    {
        return new HashMap<> (this.centresTextesTerritoires) ;
    }
    
    /**
     * Indique si les données nécessaires au tracé de la carte ont été modifiées depuis le dernier appel à
     *   cette méthode.<br>
     * Quel que soit le résultat, le booléen est remis à faux.
     * @return Vrai si les données ont été modifiées depuis la dernière fois, faux sinon.
     */
    public synchronized boolean donneesTraceCarteModifiees ()
    {
        boolean res = this.donneesTraceCarteModifiees ;
        this.donneesTraceCarteModifiees = false ;
        return res ;
    }
    
    /**
     * Note Une copie est renvoyée : l'objet peut être accédé par différents fils d'exécution.
     * @return Une copie de la table d'association territoire -> valeur limite de loupe.
     */
    public synchronized Map<Territoire,Double> loupesTextesTerritoires ()
    {
        return new HashMap<> (this.loupesTextesTerritories) ;
    }
    
}
