/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.math;

import static java.lang.Math.acos;
import static java.lang.Math.cos;
import static java.lang.Math.sin;
import static java.lang.Math.toRadians;

import java.text.DecimalFormat;

import fr.histoiremondiale.histoire.utiles.exttypes.Chaines;





/**
 * Représente un point à la surface d'une sphère.<br>
 * Les coordonnées stockent les angles (en degrés) permettant de tracer la droite interceptant
 *   la sphère au point représenté.<br>
 * Note :  La classe est immuable ; attention au code si ce n'était plus le cas (valeur de hachage, ...).
 */
public class PointSphere
{

    private double longitude ;          // Coordonnées angulaires du point
    private double latitude ;
    private double cosLo = Double.NaN,  //les lignes trigonométriques des longitudes et latitudes
                   cosLa = Double.NaN, 
                   sinLo = Double.NaN, 
                   sinLa = Double.NaN; 
    
    private Integer valHachage = null ; // Valeur de hachage (pour ne pas la recalculer)

  
    
    /**
     * Constructeur.
     * @param longitude Longitude du point.
     * @param latitude  Latitude du point.
     */
    public PointSphere (double longitude, double latitude)
    {
        this.longitude = longitude ;
        this.latitude  = latitude ;
    }
    
    
    // Représentation sous forme de chaîne
    public String toString ()
    {
        return enChaineElaboree() ;
    }
    
    
    /**
     * Renvoie une représentation de la forme : "(lo=16°, la=-179°)".
     * @return La représentation sous forme de chaîne.
     */
    public String enChaineSimple ()
    {
        
        return "(lo=" + new DecimalFormat("#0.00").format(this.longitude) + "°, " +
        		"la=" + new DecimalFormat("#0.00").format(this.latitude) + "°)" ;
    }
    
    // Renvoie une représentation de la forme : "(16° Nord, 179° Ouest)"
    public String enChaineElaboree ()
    {
        return "(" + new DecimalFormat("#0.00").format( Math.abs (this.latitude))  
                   + "° " + (this.latitude  < 0 ? "Sud"   : "Nord") + " ; " 
                   + new DecimalFormat("#0.00").format( Math.abs (this.longitude)) 
                   + "° " + (this.longitude < 0 ? "Ouest" : "Est") + ")" ;
    }
    
    
    // Egalité
    public boolean equals (Object o)
    {
        if (o == null)
            return false ;
        if (o == this)
            return true ;
        if (o.getClass() != this.getClass())
            return false ;
        
        // Comparer les attributs
        PointSphere pt = (PointSphere) o ;
        return (this.longitude == pt.longitude) &&
               (this.latitude  == pt.latitude) ;
    }
    
    
    // Hachage
    // Note performances : c'est beaucoup plus lent en effectuant une opération mathématique sur les double
    //   dans le calcul
    private static final double rapportConversionHachge = 32768.0 / 90 ;
    public int hashCode ()
    {
        if (this.valHachage == null)
        {
            // Les latitudes et longitudes sont bornées, donc on les convertit en entiers sur deux octets
            //   puis on fusionne les deux nombres
            int longitudeConvertie = new Double (this.longitude * rapportConversionHachge).intValue() ;
            int latitudeConvertie  = new Double (this.latitude  * rapportConversionHachge).intValue() ;
            this.valHachage = (longitudeConvertie << 16) | (latitudeConvertie & 0xFFFF) ;
        }
        return this.valHachage ;
    }
    
    
    // Accesseurs
    public double longitude ()
    {
        return this.longitude ;
    }
    public double latitude ()
    {
        return this.latitude ;
    }

    public double getCosLo ()
    {
        if (Double.isNaN(this.cosLo)) this.cosLo = Math.cos (Math.toRadians (this.longitude));
        return this.cosLo ;
    }


    public double getCosLa ()
    {
        if (Double.isNaN (this.cosLa)) this.cosLa = Math.cos (Math.toRadians (this.latitude));
        return this.cosLa ;
    }


    public double getSinLo ()
    {
        if (Double.isNaN(this.sinLo)) this.sinLo = Math.sin (Math.toRadians (this.longitude));
        return this.sinLo ;
    }


    public double getSinla ()
    {
        if (Double.isNaN(this.sinLa)) this.sinLa = Math.sin (Math.toRadians (this.latitude));
        return this.sinLa ;
    }


    /**
     * Ce prédicat indique si un point est à l'intérieur du cercle terrestre de centre
     * et rayon donnés (le rayon est donné sous la forme d'un cosinus donc un point est
     * à l'intérieur ssi cosArcAB(A,B) &gt;= rayon)
     * @param centre le centre du cercle
     * @param cosRayon le rayon du cercle
     * @return vrai si le point est dans le cercle limite, faux sinon
     */
    public boolean DansLeCercle(PointSphere centre, double cosRayon)
    {
    	return this.cosArcAB (centre) >= cosRayon;
    }


    /**
     * Renvoie le cosinus de l'angle de deux points sur la terre A et B : angle (OA, OB).<br>
     * O est le centre de la sphère.<br>
     * Ce cosinus peut servir à déterminer à quelle distance B est du point A, plus il est proche,
     *   plus le cosinus est proche de 1.
     * @param Pt Second point.
     * @return Le cosinus.
     */
    public double cosArcAB (PointSphere Pt)
    {
        
        double cosLatA = cos (toRadians (this.latitude())) ;
        double cosLatB = cos (toRadians (Pt.latitude())) ;
        double cosLonAB = cos(toRadians(this.longitude() - Pt.longitude()));
        double sinLatA = sin (toRadians (this.latitude())) ;
        double sinLatB = sin (toRadians (Pt.latitude())) ;
        return cosLatA * cosLatB * cosLonAB + sinLatA * sinLatB ;
    }


    /**
     * On se donne un point A (this) et le cosinus d'un angle cosA qui détermine un 
     * cercle de centre A, un point B et le cosinus d'un angle cosB qui détermine un
     * cercle de centre B. Ce prédicat renvoie Vrai si les deux cercles sont sécants
     * et Faux sinon.
     * @param cosA le premier cosinus
     * @param B le deuxième point
     * @param cosB le deuxième cosinus
     * @return vrai si les deux cercles ont une intersection non vide, faux sinon
     */
    public boolean voisin(double cosA, PointSphere B, double cosB)
    {
    	double angleAB = acos(cosArcAB(B)),
    	       angleA = acos(cosA),
    	       angleB = acos(cosB);
		return angleAB < angleA + angleB;
    }

    /**
     * On décide que deux points seront proches si l'angle qu'ils font entre eux est inférieur à une valeur donnée
     * @param B le point dont on cherche la proximité
     * @param proximite la valeur qui détermine si un point est proche ou non
     * @return vrai si B est proche de ce point (this)
     */
    public boolean proche(PointSphere B, double proximite)
    {
    	double angleAB = acos(cosArcAB(B));
 		return (angleAB < proximite);
    }
    
    /**
     * Cette méthode fait l'inverse de enChaineElaboree et enChaineSimple.
     * Elle est l'équivalent de parseDouble de la classe Double pour les PointSphere.
     * @param chaine La chaîne à lire.
     * @return Le point lu dans la chaîne.
     */
    public static PointSphere parsePointSphere (String chaine)
    {
        String S = chaine ;
        
        //System.out.println(S);
        //tout d'abord chaine élaborée ou chaine simple ?
        //si chaine simple (lo=<longitude>°, la=<latitude>°)
        //avec <longitude> et <latitude> sous la forme #0,00
        //ATTENTION la virgule décimale dépend des caractères nationaux, 
        //c'est une virgule en France
        if (S.contains ("lo="))
        {
            double longitude = 
                Double.parseDouble (Chaines.entreMarqueurs (S,"lo=", "°").replace (",", "."));
            double latitude = 
                Double.parseDouble (Chaines.entreMarqueurs (S,"la=", "°").replace (",", "."));
            return new PointSphere(longitude,latitude);
        }
        //si chaine élaborée (<latitude>° <Nord|Sud>, <longitude>° <Est|Ouest>)
        //avec <longitude> et <latitude> sous la forme #0,00 (voir ci-dessus)
        else if (S.contains ("°"))
        {
            //S = (<latitude>° <Nord|Sud>, <longitude>° <Est|Ouest>)
            //la latitude
            double latitude = 
                Double.parseDouble (Chaines.entreMarqueurs (S,"(", "°").replace (",", "."));
            //traiter la suite
            S = Chaines.apresMarqueur (S, "° ");
            //S = <Nord|Sud>, <longitude>° <Est|Ouest>)
            //le point cardinal
            String dir = Chaines.avantMarqueur (S, ",");
            //si c'est le Nord, rien à faire
            if (dir.equals ("Nord"));
            //si c'est le Sud, la latitude est négative
            else if (dir.equals ("Sud")) latitude = - latitude;
            //sinon c'est une erreur qu'il faudrait traiter
            else throw new IllegalArgumentException ("La chaîne ne correspond pas à un PointSphere (la direction devrait être Nord ou Sud) : \"" + chaine + "\"") ;
            //traiter la suite
            S = Chaines.apresMarqueur (S, ", ");
            //S = <longitude>° <Est|Ouest>) 
            //la longitude
            double longitude = 
                Double.parseDouble (Chaines.avantMarqueur (S,"°").replace (",", "."));
            //traiter la suite
            S = Chaines.apresMarqueur (S, "° ");
            //S = <Est|Ouest>) 
            //le point cardinal
            dir = Chaines.avantMarqueur (S, ")");
            //si c'est l'Est, rien à faire
            if (dir.equals ("Est"));
            //si c'est l'Ouest, la longitude est négative
            else if (dir.equals ("Ouest")) longitude = - longitude;
            //sinon c'est une erreur qu'il faudrait traiter
            else throw new IllegalArgumentException ("La chaîne ne correspond pas à un PointSphere (la direction devrait être Est ou Ouest) : \"" + chaine + "\"") ;
            
            return new PointSphere(longitude,latitude);
        }
        else
        {
            throw new IllegalArgumentException ("La chaîne ne correspond pas à un PointSphere : \"" + chaine + "\"") ;
        }
    }


	public void longitude(double d) 
	{
		longitude = d;
	}


	public void latitude(double d) 
	{
		latitude = d;
	}


	public boolean egal(PointSphere pt) 
	{
		return (pt.longitude() == this.longitude) && (pt.latitude() == this.latitude);
	}


    
}
