/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 * 
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou le modifier sous les
 * termes de la licence Affero GPL publiée par la Fondation pour le logiciel libre (Free
 * Software Foundation), en choisissant la version 3 de cette licence ou n'importe quelle
 * version ultérieure, à votre convenance.
 * 
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE D'AUCUNE SORTE
 * : y compris d'être vendable ou de pouvoir servir un but donné. Voir le texte de la
 * licence AGPL pour plus de détails.
 * 
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire. Si ce n'est pas le
 * cas, regardez à cette adresse : <http://www.gnu.org/licenses/>.
 * 
 */
package fr.histoiremondiale.histoire.utiles.math;

import static java.lang.Math.abs;
import static java.lang.Math.atan;
import static java.lang.Math.cos;
import static java.lang.Math.sin;
import static java.lang.Math.sqrt;
import static java.lang.Math.toDegrees;
import static java.lang.Math.toRadians;
import fr.histoiremondiale.histoire.igraphique.PanCarte;
import fr.histoiremondiale.histoire.utiles.exttypes.Flottants;

/**
 * Fonctions de manipulation de coordonnées à la surface d'une sphère, comme des
 * coordonnées terrestres.<br>
 * Coefficients d'ajustement : des coefficients sont utilisés dans les calculs pour que
 * l'image (carrée) représente un "carré" de 60° de côté.
 */
public class CoordSphere
{
	/**
	 * la carte représente au maximum 30° autour du centre. Cette valeur est définie dans
	 * PanCarte (plus exactement son double qui représente l'ouverture angulaire maximum
	 * entre deux points de la carte). On stocke aussi les lignes trigonométriques de cet
	 * angle.
	 */
	public static final int	ANGLE	= PanCarte.ANGLE_CARTE / 2;
	public static final double cosAngle = Math.cos(Math.toRadians(ANGLE));
	public static final double sinAngle = Math.sin(Math.toRadians(ANGLE));

	/**
	 * Renvoie la matrice de changement de coordonnées permettant de passer des
	 * coordonnées sphériques aux coordonnées dans le plan.
	 * 
	 * @param ptCentrePlan
	 *            : point à projeter au centre du plan
	 * @return La matrice de changement de coordonnées.
	 */
	public static Matrice matChgtSphereVersPlan(PointSphere ptCentrePlan)
	{
		double cosPhi; // Sinus et cosinus des angles du point central
		double sinPhi; // Phi = longitude
		double cosTeta; // Teta = latitude
		double sinTeta;
		double[][] valsMatrice; // Matrice de changement de coordonnées

		// Calculer les valeurs intermédiaires
		cosPhi = cos(toRadians(ptCentrePlan.longitude()));
		sinPhi = sin(toRadians(ptCentrePlan.longitude()));
		cosTeta = cos(toRadians(ptCentrePlan.latitude()));
		sinTeta = sin(toRadians(ptCentrePlan.latitude()));

		// Remplir la matrice
		valsMatrice = new double[][] { { cosPhi * cosTeta, -sinPhi, -cosPhi * sinTeta },
				{ sinPhi * cosTeta, cosPhi, -sinPhi * sinTeta }, { sinTeta, 0, cosTeta } };
		// Renvoyer la matrice de changement de coordonnées
		return new Matrice(valsMatrice);
	}

	/**
	 * Renvoie la matrice de changement de coordonnées permettant de passer des
	 * coordonnées sphériques aux coordonnées dans le plan.<br>
	 * Version pour la deuxième méthode de tracé.
	 * 
	 * @param ptCentrePlan
	 *            Point à projeter au centre du plan.
	 * @return La matrice de changement de coordonnées.
	 */
	public static Matrice matChgtSphereVersPlan2(PointSphere ptCentrePlan)
	{
		double cosLon; // Sinus et cosinus des angles du point central
		double sinLon; // Phi = longitude
		double cosLat; // Teta = latitude
		double sinLat;
		double[][] valsMatrice; // Matrice de changement de coordonnées

		// Calculer les valeurs intermédiaires
		cosLon = cos(toRadians(ptCentrePlan.longitude()));
		sinLon = sin(toRadians(ptCentrePlan.longitude()));
		cosLat = cos(toRadians(ptCentrePlan.latitude()));
		sinLat = sin(toRadians(ptCentrePlan.latitude()));

		// Remplir la matrice
		valsMatrice = new double[][] { { cosLon * cosLat, sinLon * cosLat, sinLat }, { -sinLon, cosLon, 0 },
				{ -cosLon * sinLat, -sinLon * sinLat, cosLat } };
		// Renvoyer la matrice de changement de coordonnées
		return new Matrice(valsMatrice);
	}

	/**
	 * Renvoie la matrice de changement de coordonnées permettant de passer des
	 * coordonnées dans le plan aux coordonnées sphériques.
	 * 
	 * @param ptCentrePlan
	 *            Point à projeter au centre du plan.
	 * @return La matrice de changement de coordonnées.
	 */
	public static Matrice matChgtPlanVersSphere(PointSphere ptCentrePlan)
	{
		return matChgtSphereVersPlan(ptCentrePlan).transformation();
	}

	/**
	 * Transforme le point de coordonnées terrestres en coordonnées dans un plan.
	 * 
	 * @param ptSphere
	 *            Point en coordonnées terrestres.
	 * @param matriceChgt
	 *            Matrice de changement de coordonnées.
	 * @param largeurPlan
	 *            Largeur du plan.
	 * @param hauteurPlan
	 *            Hauteur du plan.
	 * @param loupe
	 *            Valeur de grossissement de la loupe.
	 * @return Le point dans le plan.
	 */
	public static PointPlan coordSphereVersPlan(PointSphere ptSphere, Matrice matriceChgt, double largeurPlan,
			double hauteurPlan, double loupe)
	{
		double PX; // Coordonnées du point sur la sphère, en radians
		double PY;
		double[] ancCoord;
		double[] nouvCoord;
		double coeffX; // Coefficients d'ajustement permettant de représenter un "carré"
		// de n degrés sur la sphère par un carré dans le plan
		double coeffY;
		double coordX; // Coordonnées du point dans le plan
		double coordY;

		// Vérifier les paramètres
		double a = (largeurPlan > hauteurPlan ? largeurPlan : hauteurPlan);

		// Convertir les coordonnées sphériques du point central (?) en radians
		PX = Math.toRadians(ptSphere.longitude());
		PY = Math.toRadians(ptSphere.latitude());

		// Calculer les coefficients d'ajustement
		// On essaie avec 30 degrés
		// Ne pas oublier de faire de même pour planVersSphère
		// double angle = 30.0 ;
		double cosAngle = Math.cos(Math.toRadians(ANGLE));
		double sinAngle = Math.sin(Math.toRadians(ANGLE));
		coeffX = (a * cosAngle) / (2 * sinAngle);
		// (nouvelle formule : coeffY = coeffX)
		coeffY = coeffX;// * cos20degres ;

		// Calculer les anciennes coordonnées
		// Il s'agit d'une transformation dans l'espace, C'est le repère qui change donc
		// nouvelles coordonnées (càd : coordonnées dans le nouveau repère) =
		// anciennes coordonnées (coordonnées dans l'ancien repère) * matrice de
		// changement de repère
		ancCoord = new double[] { Math.cos(PY) * Math.cos(PX), Math.cos(PY) * Math.sin(PX), Math.sin(PY) };

		// Calculer les nouvelles coordonnées
		nouvCoord = new double[] {
				matriceChgt.val(1, 1) * ancCoord[0] + matriceChgt.val(2, 1) * ancCoord[1] + matriceChgt.val(3, 1)
						* ancCoord[2],
				matriceChgt.val(1, 2) * ancCoord[0] + matriceChgt.val(2, 2) * ancCoord[1] + matriceChgt.val(3, 2)
						* ancCoord[2],
				matriceChgt.val(1, 3) * ancCoord[0] + matriceChgt.val(2, 3) * ancCoord[1] + matriceChgt.val(3, 3)
						* ancCoord[2] };

		// Ajuster les nouvelles coordonnées
		if (nouvCoord[0] < 0.00001)
		{
			nouvCoord[0] = 0.00001;
		}
		coordX = coeffX * loupe * nouvCoord[1] / nouvCoord[0];
		coordY = -coeffY * loupe * nouvCoord[2] / nouvCoord[0];

		// Renvoyer le point en coordonnées du plan
		return new PointPlan(coordX, coordY);
	}

	/**
	 * Transforme le point du plan en coordonnées terrestres.<br>
	 * Attention : la matrice de changement de coordonnées n'est pas la même que pour
	 * l'opération inverse.<br>
	 * Note : Cette fonction ne fait pas l'opération inverse de la précédente, elle inclut
	 * l'ajustement par rapport au centre du plan.
	 * 
	 * @param ptPlan
	 *            Point du plan.
	 * @param ptCentralSphere
	 *            Point projeté au centre du plan.
	 * @param matriceChgt
	 *            Matrice de changement de coordonnées.
	 * @param largeurPlan
	 *            Largeur du plan.
	 * @param hauteurPlan
	 *            Hauteur du plan.
	 * @param loupe
	 *            Valeur de grossissement de la loupe.
	 * @return Le point de la sphère correspondant.
	 */
	public static PointSphere coordPlanVersSphere(PointPlan ptPlan, PointSphere ptCentralSphere, Matrice matriceChgt,
			double largeurPlan, double hauteurPlan, double loupe)
	{
		double PX;
		double PY;
		double ancCoordX;
		double ancCoordY;
		double ancCoordZ;
		double[] nouvCoord;
		double coeffX; // Coefficients d'ajustement permettant de représenter un "carré"
		// de n degrés sur la sphère par un carré dans le plan
		double coeffY;
		double longitude;
		double latitude;

		// Vérifier les paramètres
		// (la largeur et la hauteur sont censées être identiques)
		double a = (largeurPlan > hauteurPlan ? largeurPlan : hauteurPlan);
		// if (! Flottants.egaux (largeurPlan, hauteurPlan))
		// throw new IllegalArgumentException
		// ("La largeur et la hauteur devraient être égales : " + largeurPlan + " et " +
		// hauteurPlan) ;

		// Calculer les coefficients d'ajustement
		// on essaie avec 30 degrés
		// double angle = 30.0 ;
		double cosAngle = cos(toRadians(ANGLE));
		double sinAngle = sin(toRadians(ANGLE));
		coeffX = (a * cosAngle) / (2 * sinAngle);
		// coeffX = (largeurPlan * cos20degres) / (2 * sin20degres) ;
		// (nouvelle formule : coeffY=coeffX)
		// coeffY = coeffX * cos20degres ;
		coeffY = coeffX;

		// Calcul des coordonnées 0 (?)
		PX = (ptPlan.x() - largeurPlan / 2) / (loupe * coeffX);
		PY = (hauteurPlan / 2 - ptPlan.y()) / (loupe * coeffY);
		/*
		 * Ne pas inclure l'ajustement OK pour le centrage, erreur pour les textes PX =
		 * ptPlan.x() / (loupe * coeffX) ; PY = ptPlan.y() / (loupe * coeffY) ;
		 */
		/*
		 * PX = (ptPlan.x() - a / 2) / (loupe * coeffX) ; PY = (a / 2 - ptPlan.y()) /
		 * (loupe * coeffY) ;
		 */
		// Calcul des anciennes et nouvelles coordonnées (?)
		ancCoordX = 1.0 / (sqrt(1 + PX * PX + PY * PY));
		ancCoordY = PX * ancCoordX;
		ancCoordZ = PY * ancCoordX;
		nouvCoord = new double[] {
				matriceChgt.val(1, 1) * ancCoordX + matriceChgt.val(2, 1) * ancCoordY + matriceChgt.val(3, 1)
						* ancCoordZ,
				matriceChgt.val(1, 2) * ancCoordX + matriceChgt.val(2, 2) * ancCoordY + matriceChgt.val(3, 2)
						* ancCoordZ,
				matriceChgt.val(1, 3) * ancCoordX + matriceChgt.val(2, 3) * ancCoordY + matriceChgt.val(3, 3)
						* ancCoordZ };

		// Calcul de la longitude et de la latitude
		if (Flottants.arrondir(abs(nouvCoord[2]), 5) == 1)
		{
			longitude = ptCentralSphere.longitude();
			latitude = (nouvCoord[2] > 0 ? 90 : -90);
		}
		else if (Flottants.arrondir(nouvCoord[0], 5) == 0)
		{
			longitude = (nouvCoord[1] > 0 ? 90 : -90);
			latitude = Flottants.arrondir(toDegrees(atan(nouvCoord[2] / abs(nouvCoord[1]))), 2);
		}
		else
		{
			longitude = Flottants.arrondir(toDegrees(atan(nouvCoord[1] / nouvCoord[0])), 2);
			longitude += (nouvCoord[0] > 0 ? 0 : 180);
			if (longitude > 180)
			{
				longitude -= 360;
			}
			latitude = Flottants.arrondir(toDegrees(atan(cos(toRadians(longitude)) * nouvCoord[2] / nouvCoord[0])), 2);
		}

		// Renvoyer le point calculé
		return new PointSphere(longitude, latitude);
	}

	/**
	 * calcule la projection du point sur le plan (origine au centre de l'écran)
	 * 
	 * @param point
	 *            le point que l'on veut projeter
	 * @param matrice
	 *            la matrice qui permet de changer d ecoordonnées dans l'espace
	 * @param loupe
	 *            la loupe de la carte
	 * @param coteCarre
	 *            le max de la largeur et la hauteur de la carte
	 * @return la projection du point sur le plan
	 */
	public static PointPlan projection(PointSphere point, Matrice matrice, double loupe, int coteCarre)
	{
		PointEspace ptEspace = CalculsEspace.ptEspaceDepuisPtSphere(point);
		PointEspace ptEspaceNouvCoord = CalculsEspace.multiplier(matrice, ptEspace);
		PointPlan ptPlan = CalculsEspace.projection(ptEspaceNouvCoord, loupe, coteCarre);
		return ptPlan;
	}

	/**
	 * Renvoie le point de la terre correspondant au point de l'écran
	 * 
	 * @param P
	 *            un point de l'écran
	 * @param centre
	 *            le centre de la carte
	 * @param largeurCarte
	 *            la largeur de la carte
	 * @param hauteurCarte
	 *            la hauteur de la carte
	 * @param loupe
	 *            la loupe de la carte
	 * @return le point de la terre correspondant
	 */
	public static PointSphere InvProjection(PointPlan P, PointSphere centre, int largeurCarte, int hauteurCarte,
			double loupe)
	{
		/*
		 * passer du repère de l'écran au repère centré sur le coin supérieur gauche de la
		 * carte (axe des abscisses vers la droite, axe des ordonnées vers le haut) H est
		 * la hauteur du panneau d'affichage de la carte, L sa largeur a est le côté du
		 * carré de la carte (c'est la plus grande des dimension du panneau) a = max(H,L)
		 * X(centre carte) = X(origine écran) + (a - L)/2 Y(centre carte) = Y(origine
		 * écran) + (a - H)/2 X(origine écran) = Y(origine écran) = 0
		 */
		double a = Math.max(hauteurCarte, largeurCarte), 
		       X = P.x() + (a - largeurCarte) / 2, 
		       Y = P.y() + (a - hauteurCarte) / 2;

		/*
		 * passage du repère du coin supérieur gauche de la carte (origine en haut à
		 * gauche, axe des abscisses vers la droite, axe des ordonnées vers le bas) au
		 * repère dans le plan tangent à la terre (voir documentation) (origine au centre
		 * de la carte, axe des abscisses vers la droite, axe des ordonnées vers le haut)
		 * x' = (X - L/2)/(loupe * coefftX) y' = (H/2 - Y)/(loupe * coefftY) CoefftX = a /
		 * (2*tan(30°)) = a / 1.154700538 = a / COEFFT_CARTE CoefftY = CoefftX
		 */
		double coefftX = a / PanCarte.COEFFT_CARTE, 
		       xprime = (X - a / 2) / (loupe * coefftX), 
		       yprime = (a / 2 - Y) / (loupe * coefftX);

		/*
		 * passage du repère centré sur la carte au repère dans l'espace (axes des
		 * coordonnées liés au point central) x" = 1/(Racine carrée de (1 + x’² + y’²)
		 * y" = x’ * x" z" = y’ * x"
		 */
		double xseconde = 1 / Math.sqrt(1 + xprime * xprime + yprime * yprime), 
		       yseconde = xprime * xseconde, 
		       zseconde = yprime * xseconde;

		/*
		 * passage du repère lié au point central au repère normalisé dans l'espace
		 * centré, axes des coordonnées liés à l'équateur et au méridien de Greenwich x =
		 * cos(λ0) cos(θ0) * x" - sin(λ0) * y" - cos(λ0) sin(θ0) * z" y = sin(λ0) cos(θ0)
		 * * x" + cos(λ0) y" - sin(λ0) sin(θ0) * z" z = sin(θ0) * x" + 0 * y" + cos(θ0) *
		 * z" λ0 = longitude du point central, θ0 = latitude du point central;
		 */
		double x = centre.getCosLo() * centre.getCosLa() * xseconde - centre.getSinLo() * yseconde 
		         - centre.getCosLo() * centre.getSinla() * zseconde, 
			   y = centre.getSinLo() * centre.getCosLa() * xseconde + centre.getCosLo() * yseconde 
			     - centre.getSinLo() * centre.getSinla() * zseconde, 
			   z = centre.getSinla() * xseconde + centre.getCosLa() * zseconde;

		/*
		 * passage du repère dans l'espace aux coordonnées géographiques z = 1 : longitude
		 * = longitude du point central, latitude = 90° z = -1 : longitude = longitude du
		 * point central, latitude = -90° x = 0 et y > 0 : longitude = 90°, latitude =
		 * arcsinus(z) x = 0 et y < 0 : longitude = -90°, latitude = arcsinus(z) x > 0 :
		 * longitude = arctangente(y/x), latitude = arcsinus(z) x < 0 : longitude = PI +
		 * arctangente(y/x), latitude = arcsinus(z) si longitude > 180 alors longitude =
		 * longitude - 360;
		 */
		double latitude = toDegrees(Math.asin(z)), longitude;
		if (Math.abs(z) == 1)
		{
			longitude = centre.longitude();
		}
		else if (x == 0)
		{
			longitude = (y > 0 ? 90 : -90);
		}
		else if (x > 0)
		{
			longitude = toDegrees(Math.atan(y / x));
		}
		else if (x < 0)
		{
			longitude = 180 + toDegrees(Math.atan(y / x));
		}
		else
		{
			longitude = 0;
		}

		if (longitude > 180)
		{
			longitude = longitude - 360;
		}
		return new PointSphere(longitude, latitude);
	}

}
