/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.utiles.igraphique.dessin.hachures;

import java.awt.Color ;
import java.awt.PaintContext ;
import java.awt.geom.AffineTransform ;
import java.awt.geom.Point2D ;
import java.awt.image.ColorModel ;
import java.awt.image.DataBufferInt ;
import java.awt.image.Raster ;



/**
 * Contexte de couleurs pour la peinture de hachures.<br>
 * Note : Les transformations du contexte graphique ne sont pas prises en compte, ce qui rend les hachures
 *        invariantes à des opérations de rotation ou d'échelle, pour le meilleur et pour le pire.
 */
public class ContextePeintureHachures implements PaintContext
{

    private static ColorModel modeleCouleurs = ColorModel.getRGBdefault() ;
    
    // Paramètre des hachures
    private Color  couleur ;            // Couleur
    private double espace ;             // Espace entre les hachures
    private double epaisseur ;          // Epaisseur
    
    // Attributs pour les traitements
    private int[] tableCouleurs ;       // Table des couleurs (dégradés pour créer les hachures)
    private int   largeurFlou ;         // Largeur de la zone couverte par une hachure
    private AffineTransform transf ;    // Transformation de position du gradient (?)
    

    
    /**
     * Constructeur.
     * @param couleur     La couleur des hachures.
     * @param espace      L'espace entre les hachures.
     * @param angle       L'angle des hachures.
     * @param epaisseur   L'épaisseur des hachures.
     * @param largeurFlou Largeur du flou autour des hachures (?).
     */
    public ContextePeintureHachures (Color couleur, double espace, double angle, double epaisseur, int largeurFlou)
    {
        // Enregistrement des paramètres
        this.couleur     = couleur ;
        this.espace      = espace ;
        this.epaisseur   = epaisseur ;
        this.largeurFlou = largeurFlou ;

        // Initialisation des autres attributs
        this.tableCouleurs = creerTableCouleurs (couleur, largeurFlou) ;
        this.transf        = new AffineTransform() ;
        this.transf.rotate (Math.PI / 2 + angle) ;          // Décalage de PI/2 pour que l'angle 0 soit à l'horizontale
    }
    
    
    /**
     * Modifie la couleur des hachures.
     * @param nouvCouleur La nouvelle couleur.
     */
    public void modifCoulHachures (Color nouvCouleur)
    {
        if (! this.couleur.equals (nouvCouleur))
        {
            this.couleur       = nouvCouleur ;
            this.tableCouleurs = creerTableCouleurs (this.couleur, this.largeurFlou) ;
        }
    }
    
    
    /**
     * Crée la table des couleurs en fonction des paramètres de hachures donnés.<br>
     * Au centre la couleur nette, puis dégradé vers les bords (anticrénelage).
     * @param couleur     La couleur des hachures.
     * @param largeurFlou La largeur de la zone floue.
     * @return La table des couleurs.
     */
    private int[] creerTableCouleurs (Color couleur, int largeurFlou)
    {
        int   rvbCoul ;         // Composantes RVB de la couleur en paramètre (sans alpha)
        int   alphaCoul ;       // Composants alpha (transparence) de la couleur en paramètre
        int[] tableCouleurs ;   // Table des couleurs à créer
        
        // Initialisations
        rvbCoul   = couleur.getRGB() & 0xFFFFFF ;
        alphaCoul = couleur.getAlpha() ;
        tableCouleurs = new int[largeurFlou] ;
        
        // Création de la table, symétrique
        for (int i = 0 ; i < largeurFlou / 2 ; i++)
        {
            int alpha = (int) ((double)i / (largeurFlou / 2 - 1) * alphaCoul) ;
            alpha <<= 24 ;
            tableCouleurs[i] = alpha | rvbCoul ;
            tableCouleurs[tableCouleurs.length-i-1] = alpha | rvbCoul ;
        }
        
        // Remplir le centre si taille impaire
        if (largeurFlou % 2 != 0)
            tableCouleurs[largeurFlou/2] = (alphaCoul << 24) | rvbCoul ;
        
        // Renvoyer la table des couleurs
        return tableCouleurs ;
    }
    

    // Renvoie un échantillon du rendu de la surface à peindre (?)
    public Raster getRaster (int x, int y, int largeur, int hauteur)
    {
        Raster        echantillon ;         // Echantillon (?) à renvoyer
        DataBufferInt donneesEch ;          // Données de l'échantillon à renvoyer
        int[]         points ;              // Points de l'échantillon à renvoyer
        
        
        // Récupérer le tampon (échantillon) à peindre
        echantillon = modeleCouleurs.createCompatibleWritableRaster (largeur, hauteur) ;
        donneesEch  = (DataBufferInt) echantillon.getDataBuffer() ;
        points      = donneesEch.getData() ;

        // Autres initialisations
        double xRef       = 0 ;
        double distance   = this.espace ;
        int decalage      = donneesEch.getOffset() ;
        int ajustement    = echantillon.getWidth() - largeur ;
        Point2D.Double p1 = new Point2D.Double() ;      // Coordonnées dans l'image du pixel actuel
        Point2D.Double p2 = new Point2D.Double() ;      // Coordonnées dans le repère des hachures verticales

        
        // Peindre l'échantillon
        for (int iHauteur = 0 ; iHauteur < hauteur ; iHauteur++)
        {
            for (int iLargeur = 0 ; iLargeur < largeur ; iLargeur++)
            {
                double ligne ;
                double r = 0 ;
                p1.x = x + iLargeur ;
                p1.y = y + iHauteur ;
                this.transf.transform (p1, p2) ;
                
                // Calcul de la distance qui sépare le point de la plus proche hachure verticale
                r = (p2.x - xRef) / distance ;
                r = r - Math.floor (r) ;
                ligne = Math.abs (r * distance) ;

                // Peindre si nécessaire (trait ° zone floue)
                points[decalage++] = (ligne  < this.epaisseur ?
                                        this.tableCouleurs[(int)(ligne / this.epaisseur * this.largeurFlou)] :
                                        0) ;
            }
            decalage += ajustement ;
        }
        
        // Renvoyer l'échantillon
        return echantillon ;
    }
    
    
    // Renvoie le modèle de couleurs utilisé
    public ColorModel getColorModel ()
    {
        return modeleCouleurs ;
    }
    
    
    // Libérer les ressources utilisées
    public void dispose()
    {
    }
    
}
