/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.stockage;

import java.awt.Color ;
import java.sql.ResultSet ;
import java.sql.SQLException ;
import java.util.ArrayList ;
import java.util.HashMap ;
import java.util.HashSet ;
import java.util.List ;
import java.util.Map ;

import fr.histoiremondiale.histoire.donnees.CentreCivilisation;
import fr.histoiremondiale.histoire.donnees.Civilisation;
import fr.histoiremondiale.histoire.donnees.Limes;
import fr.histoiremondiale.histoire.donnees.LimesOrdonne;
import fr.histoiremondiale.histoire.donnees.Pourtour;
import fr.histoiremondiale.histoire.donnees.Territoire;
import fr.histoiremondiale.histoire.donnees.TexteTerritoire;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;



/**
 * Classe d'accès aux données historiques.
 * 
 * Modif 15/07/2019 : Si une civilisation n'a pas de centre elle n'est pas prise en compte 
 * dans la liste des civilisations. C'est le cas par exemple si cette civilisation n'a pas
 * encore de centre défini (travail encours). Pour qu'elle soit quand même pris en compte 
 * il faut utiliser un LEFT JOIN au lieu d'un INNER JOIN.
 */
public class AccesDonneesHistoriques extends AccesDonneesBase
{
    
    
    /**
     * Constructeur.
     * @param chemRepDonnees Chemin du répertoire d'accès aux données.
     */
    public AccesDonneesHistoriques (String chemRepDonnees)
    {
        super (chemRepDonnees) ;
    }
    
    
    /**
     * Renvoie la liste des civilisations, triée par ordre alphabétique.
     * @return La liste des civilisations.
     */
    public List<Civilisation> chargerCivilisations ()
    {
        try
        {
            String txtRequete = "SELECT civilisations.*, centres_civilisations.*" +
                                "  FROM civilisations" +
                                "  LEFT JOIN centres_civilisations ON civilisations.id = centres_civilisations.id_civ" +
                                " ORDER BY civilisations.nom" ;

            // Charger les données
            List<Civilisation> civilisations = new ArrayList<>() ;
            ResultSet          res           = connexionCourante().prepareStatement(txtRequete).executeQuery() ;
            Civilisation       civCourante   = null ;
            while (res.next())
            {
                // Lire la ligne
                int    idCivilisation  = res.getInt    ("civilisations.id") ;
                String nomCivilisation = res.getString ("civilisations.nom") ;
                double longitudeCentre = res.getDouble ("centres_civilisations.longitude") ;
                double latitudeCentre  = res.getDouble ("centres_civilisations.latitude") ;
                int    debutCentre     = res.getInt    ("centres_civilisations.debut") ;
                int    finCentre       = res.getInt    ("centres_civilisations.fin") ;
                
                // Créer un nouvel objet si nécessaire
                if (civCourante == null || idCivilisation != civCourante.id())
                {
                    civCourante = new Civilisation (idCivilisation, nomCivilisation) ;
                    civilisations.add (civCourante) ;
                }
                
                // Collecter les centres de la civilisation
                civCourante.centres().add (new CentreCivilisation (longitudeCentre,
                                                                   latitudeCentre,
                                                                   debutCentre,
                                                                   finCentre)) ;
            }
            
            // Fermer les objets de résultat
            res.close() ;
            
            // Renvoyer le résultat
            return civilisations ;
        }
        catch (SQLException e)
        {
            throw new RuntimeException ("Erreur lors du chargement des données", e) ;
        }
    }

    
    /**
     * Renvoie la liste des territoires, triés par date de début.
     * @param indexCivilisationsParId Table d'association id -> civilisation.
     * @param indexLimes              Table d'association id -> limes.
     * @return La liste des territoires.
     */
    public List<Territoire> chargerTerritoires (Map<Integer,Civilisation> indexCivilisationsParId, Map<Integer,Limes> indexLimes)
    {
        // Requêtes pour les pourtours
        String txtRequeteIdPourtourMax  = "SELECT MAX (id)" +
                                          "  FROM pourtours" ;
        String txtRequetePourtours      = "SELECT id, long_centre, lat_centre, cos_cercle_limite" +
                                        "  FROM pourtours" ;                // (on charge tous les pourtours, mais en pratique la plupart concernent des territoires et on évite une jointure)
        String txtRequeteLimesPourtours = "SELECT id_pourtour, id_limes, sens" +
                                          "  FROM pourtours_limes" +
                                          " ORDER BY id_pourtour, rang" ;   // (pour garder les limes dans l'ordre)
        
        // Requêtes pour les territoires
        // Note performances : en redécoupant cette requête en deux on obtient une requête un peu plus rapide
        //                     mais ça se joue à pas grand chose et il faut faire plus d'extraction et plus
        //                     d'accès à la table de hachage indexant les territoires, ce qui devrait boucher
        //                     le faible écart.
        String txtRequeteTerritoires      = "SELECT territoires.id," +
                                            "       territoires.rang," +
                                            "       noms_territoires.nom," +
                                            "       territoires.id_civ," +
                                            "       territoires.couleur_interieur," +
                                            "       territoires.couleur_frontiere," +
                                            "       territoires.hachure," +
                                            "       territoires.style," +
                                            "       territoires.debut," +
                                            "       territoires.fin," +
                                            "       territoires.val_loupe_limite_affichage," +
                                            "       territoires.id_pourtour" +
                                            "  FROM territoires" +
                                            "       LEFT JOIN noms_territoires  ON territoires.id_nom = noms_territoires.id" +
                                            " ORDER BY territoires.rang" ;
        String txtRequeteInfosTerritoires = "SELECT id_territoire, debut, fin, texte, italique" +
                                            "  FROM infos_territoires" ;

        
        try
        {
    
            
            // Charger les pourtours
            ResultSet resIdPourtourMax = connexionCourante().prepareStatement(txtRequeteIdPourtourMax).executeQuery() ;
            ResultSet resPourtour      = connexionCourante().prepareStatement(txtRequetePourtours).executeQuery() ;
            ResultSet resLimesPourtour = connexionCourante().prepareStatement(txtRequeteLimesPourtours).executeQuery() ;
            // (pourtours)
            resIdPourtourMax.next() ;
            int idPourtourMax = resIdPourtourMax.getInt (1) ;
            Pourtour[] pourtours = new Pourtour[idPourtourMax+1] ;
            while (resPourtour.next())
            {
                int idPourtour = resPourtour.getInt (1) ;
                pourtours[idPourtour] = new Pourtour (idPourtour,
                                                      new ArrayList<>(),
                                                      new PointSphere (resPourtour.getDouble (2),
                                                                       resPourtour.getDouble (3)),
                                                      resPourtour.getDouble (4)) ;
            }
            // (limes des pourtours)
            while (resLimesPourtour.next())
            {
                int     idPourtour   = resLimesPourtour.getInt     (1) ;
                int     idLimes      = resLimesPourtour.getInt     (2) ;
                boolean sensLimes    = resLimesPourtour.getBoolean (3) ;
                pourtours[idPourtour].limesOrdonnes().add (new LimesOrdonne (indexLimes.get (idLimes), sensLimes)) ;
            }
            
            // Fermer les objets de résultat
            resIdPourtourMax.close() ;
            resPourtour.close() ;
            resLimesPourtour.close() ;
        

            // Charger les territoires
            List<Territoire>        territoires           = new ArrayList<>() ;
            Map<Integer,Territoire> indexTerritoiresParId = new HashMap<>() ;
            // (territoires)
            ResultSet resTerr      = connexionCourante().prepareStatement(txtRequeteTerritoires).executeQuery() ;
            ResultSet resInfosTerr = connexionCourante().prepareStatement(txtRequeteInfosTerritoires).executeQuery() ;
            while (resTerr.next())
            {
                Territoire territoire = new Territoire (resTerr.getInt    (1),
                                                        resTerr.getInt    (2),
                                                        resTerr.getString (3),
                                                        new HashSet<TexteTerritoire>(),
                                                        indexCivilisationsParId.get (resTerr.getInt (4)),
                                                        new Color (resTerr.getInt (5), true),
                                                        new Color (resTerr.getInt (6)),
                                                        resTerr.getInt    (7),
                                                        resTerr.getInt    (8),
                                                        resTerr.getInt    (9),
                                                        resTerr.getInt    (10),
                                                        resTerr.getInt    (11),
                                                        pourtours[resTerr.getInt (12)]) ;
                territoires.add (territoire) ;
                indexTerritoiresParId.put (territoire.id(), territoire) ;
            }
            // (textes des territoires)
            while (resInfosTerr.next())
            {
                int             idTerritoire = resInfosTerr.getInt (1) ;
                Territoire      territoire   = indexTerritoiresParId.get (idTerritoire) ;
                TexteTerritoire texte        = new TexteTerritoire (resInfosTerr.getInt     (2),
                                                                    resInfosTerr.getInt     (3),
                                                                    resInfosTerr.getString  (4),
                                                                    resInfosTerr.getBoolean (5)) ;
                territoire.textes().add (texte) ;
            }

            // Fermer les objets de résultat
            resTerr.close() ;
            resInfosTerr.close() ;
            
            // Renvoyer le résultat
            return territoires ;
        }
        catch (SQLException e)
        {
            throw new RuntimeException ("Erreur lors du chargement des données", e) ;
        }
    }
    
    
}
