/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.stockage;

import static fr.histoiremondiale.histoire.igraphique.FenPrincipale.PROPORTION_SEPARATEUR_DEFAUT;
import static fr.histoiremondiale.histoire.utiles.es.fichiers.Proprietes.chargerProprietes;
import static fr.histoiremondiale.histoire.utiles.es.fichiers.Proprietes.sauverProprietes;

import java.io.File ;
import java.io.IOException ;
import java.util.Properties ;

import fr.histoiremondiale.histoire.ConfigAppli;
import fr.histoiremondiale.histoire.DonneesHistoriques;
import fr.histoiremondiale.histoire.EtatAppli;
import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.igraphique.FenPrincipale;



/**
 * Outil de chargement/sauvegarde de la configuration de l'appli.
 */
public class AccesDonneesConfiguration
{
 
    private static final String ENCODAGE_FIC_CONFIG = "UTF-8" ;
    private static final String CHEM_FIC_CONFIG     = System.getProperty ("user.home") + File.separator +
                                                      ".config/histoiremondiale/histoire.conf".replace ('/', File.separatorChar) ;
    
    
    /**
     * Sauvegarde la configuration de l'appli.
     * @param application Instance de l'applcation.
     * @throws IOException En cas d'erreur d'accès aux fichiers.
     */
    public void sauverConfig (HistoireMondiale application) throws IOException
    {
        EtatAppli     etatAppli = application.etat() ;
        FenPrincipale fenPpale  = application.fenPrincipale() ;
        
        
        // Stocker les valeurs dans une liste de propriétés
        Properties props = new Properties() ;
        // (ne sauver le chemin que s'il a été fourni explicitement dans le fichier de configuration, pas si
        //  l'a récupéré dans une variable d'environnement ou si on a pris le chemin par défaut ; notamment
        //  si on utilise plusieurs applis lancées avec des variables d'environnement différentes ou si on
        //  ne veut pas modifier le fichier en lançant une autre version de l'appli ; le chemin est réécrit
        //  tel que lu dans le fichier de configuration)
        if (application.config().chemRepDonneesLuConf() != null)
            props.put ("chemRepDonnees",              application.config().chemRepDonneesLuConf()) ;
        props.put ("longitude",                     ""+etatAppli.longitude()) ;
        props.put ("latitude",                      ""+etatAppli.latitude()) ;
        // on sauve la valeur de la loupe et non son indice
        //props.put ("iLoupe",                  ""+etatAppli.iLoupe()) ;
        props.put ("loupe",                         ""+etatAppli.loupe ()) ;
        props.put ("annee",                         ""+etatAppli.annee()) ;
        props.put ("nomRegionCourante",             (etatAppli.civilisationSelectionnee() == null ? "" : etatAppli.civilisationSelectionnee().nom())) ;
        props.put ("verrouillageRegionActif",       ""+etatAppli.suiviCivilisationSelectionnee()) ;
        props.put ("symboleAttenteTracer",          ""+etatAppli.symboleAttenteTracer()) ;
        props.put ("texteSurvolFondClair",          ""+etatAppli.texteSurvolFondClair()) ;
        props.put ("posSeparateurCarteNavig",       ""+fenPpale.posSeparateurPanneaux()) ;
        
        props.put ("simpleClicFermerParagraphes",   ""+etatAppli.simpleClicFermerParagraphes()) ;
        props.put ("moletteGlisseDeUn",             ""+etatAppli.moletteGlisseDeUn()) ;
        props.put ("affParagraphesHtmlGeneres",     ""+etatAppli.affParagraphesHtmlGeneres()) ;
        
        props.put ("deplAnnees",                    ""+etatAppli.deplAnnees()) ;
        props.put ("deplCoordonnees",               ""+etatAppli.deplEspace()) ;
        props.put ("fleuvesVisibles",               ""+etatAppli.fleuvesAffiches()) ;
        props.put ("infobullesCarte",               ""+etatAppli.infobullesCarte()) ;
        props.put ("meridiensEtParallelesAffiches", ""+etatAppli.meridiensParallelesAffiches()) ;
        
        if (etatAppli.chemRepSauvCartes() != null)
            props.put ("chemRepSauvCartes",         etatAppli.chemRepSauvCartes()) ;
        if (etatAppli.chemRepSauvParagraphes() != null)
            props.put ("chemRepSauvParagraphes",    etatAppli.chemRepSauvParagraphes()) ;
        if (etatAppli.chemRepVueParagraphes() != null)
            props.put ("chemRepVueParagraphes",    etatAppli.chemRepVueParagraphes()) ;

        // Sauvegarder les valeurs
        sauverProprietes (props, "Configuration Histoire mondiale", CHEM_FIC_CONFIG, ENCODAGE_FIC_CONFIG) ;
    }

    
    /**
     * Charge et renvoie la configuration générale de l'appli.
     * @return La configuration de l'application.
     * @throws IOException En cas d'erreur d'accès aux fichiers.
     */
    public ConfigAppli chargerConfigGenerale () throws IOException
    {
        // Charger les données de configuration
        Properties props = chargerProprietes (CHEM_FIC_CONFIG, ENCODAGE_FIC_CONFIG) ;
        
        // Copier les informations dans l'instance de l'application
        String chemRepDonneesLuConf = lireChaine  (props, "chemRepDonnees", null) ;
        String chemRepDonnees       = (chemRepDonneesLuConf == null ?
                                         "donnees" :
                                         chemRepDonneesLuConf.replace ('/', File.separatorChar)) ;
        return new ConfigAppli (chemRepDonnees, chemRepDonneesLuConf) ;
    }
    
    
    /**
     * Charge l'état de l'appli.<br>
     * L'état est créé et renvoyé.
     * @param donneesHisto Données historiques.
     * @return L'état de l'application chargé.
     * @throws IOException En cas d'erreur d'accès aux fichiers.
     */
    public EtatAppli chargerEtatAppli (DonneesHistoriques donneesHisto) throws IOException
    {
        // Charger les données de configuration
        Properties props = chargerProprietes (CHEM_FIC_CONFIG, ENCODAGE_FIC_CONFIG) ;
        
        // Créer l'objet représentant l'état de l'application
        EtatAppli etatAppli = new EtatAppli() ;
        etatAppli.modifLongitude                     (lireDouble  (props, "longitude",                     20)) ;
        etatAppli.modifLatitude                      (lireDouble  (props, "latitude",                      40)) ;
        etatAppli.modifLoupe                         (lireDouble  (props, "loupe",                         0)) ;
        etatAppli.modifAnnee                         (lireEntier  (props, "annee",                         1)) ;
        etatAppli.modifSuiviCivilisationSelectionnee (lireBoolean (props, "verrouillageRegionActif",       true)) ;
        etatAppli.modifSymboleAttenteTracer          (lireBoolean (props, "symboleAttenteTracer",          true)) ;
        etatAppli.modifTexteSurvolFondClair          (lireBoolean (props, "texteSurvolFondClair",          false)) ;
        etatAppli.modifSimpleClicFermerParagraphes   (lireBoolean (props, "simpleClicFermerParagraphes",   false)) ;
        etatAppli.modifPosSeparateurCarteNavig       (lireDouble  (props, "posSeparateurCarteNavig",       PROPORTION_SEPARATEUR_DEFAUT)) ;
        etatAppli.modifMoletteGlisseDeUn             (lireBoolean (props, "moletteGlisseDeUn",             true)) ;
        etatAppli.modifAffParagraphesHtmlGeneres     (lireBoolean (props, "affParagraphesHtmlGeneres",     false)) ;
        etatAppli.modifDeplAnnees                    (lireEntier  (props, "deplAnnees",                    1)) ;
        etatAppli.modifDeplEspace                    (lireEntier  (props, "deplCoordonnees",               1)) ;
        etatAppli.modifFleuvesAffiches               (lireBoolean (props, "fleuvesVisibles",               true)) ;
        etatAppli.modifInfobullesCarte               (lireBoolean (props, "infobullesCarte",               true)) ;
        etatAppli.modifMeridiensParallelesAffiches   (lireBoolean (props, "meridiensEtParallelesAffiches", false)) ;
        String nomCivilisationCourante =              lireChaine  (props, "nomRegionCourante",             null) ;
        etatAppli.modifCivilisationSelectionnee      (donneesHisto.civilisationDeNom (nomCivilisationCourante)) ;
        etatAppli.modifChemRepSauvCartes             (lireChaine  (props, "chemRepSauvCartes",             null)) ;
        etatAppli.modifChemRepSauvParagraphes        (lireChaine  (props, "chemRepSauvParagraphes",        null)) ;
        etatAppli.modifChemRepVueParagraphes         (lireChaine  (props, "chemRepVueParagraphes",         null)) ;
        
        // Renvoyer l'état de l'appli
        return etatAppli ;
    }
    
    
    // Renvoie la valeur d'une propriété sous la forme d'une chaîne
    // La valeur par défaut est renvoyée si la propriété n'existe pas dans le fichier
    private String lireChaine (Properties props, String nomProp, String valDefaut)
    {
        String val = (String) props.get (nomProp) ;
        return (val == null || val.isEmpty() ? valDefaut : val.trim()) ;
    }
    
    
    // Renvoie la valeur d'une propriété sous la forme d'un entier
    // La valeur par défaut est renvoyée si la propriété n'existe pas dans le fichier
    private int lireEntier (Properties props, String nomProp, int valDefaut)
    {
        String val = (String) props.get (nomProp) ;
        return (val == null || val.isEmpty() ? valDefaut : Integer.parseInt (val.trim())) ;
    }
    
    
    // Renvoie la valeur d'une propriété sous la forme d'un booléen
    // La valeur par défaut est renvoyée si la propriété n'existe pas dans le fichier
    private boolean lireBoolean (Properties props, String nomProp, boolean valDefaut)
    {
        String val = (String) props.get (nomProp) ;
        return (val == null || val.isEmpty() ? valDefaut : Boolean.parseBoolean (val.trim())) ;
    }
    
    // Renvoie la valeur d'une propriété sous la forme d'un double
    // La valeur par défaut est renvoyée si la propriété n'existe pas dans le fichier
    private double lireDouble (Properties props, String nomProp, double valDefaut)
    {
        String val = (String) props.get (nomProp) ;
        return (val == null || val.isEmpty() ? valDefaut : Double.parseDouble (val.trim())) ;
    }
    
    
}
