/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique.composants;

import java.util.Observable ;
import java.util.Observer ;

import javax.swing.JScrollBar ;

import fr.histoiremondiale.histoire.utiles.exttypes.Objets;



/**
 * Barre de défilement observant et représentant la valeur de l'attribut d'un objet.<br>
 * La valeur de ce composant est celle sélectionnée par le curseur.
 */
public class BarreDefilement extends JScrollBar implements Observer, ConteneurValeur
{

    // Constantes
    public static final String MARQUEUR_REMPL_VAL_ENTIER   = "##val##" ;    // A remplacer par la valeur de l'attribut
    
    
    // Données
    private Object  objetObserve ;          // Objet contenant l'attribut observé
    private String  nomAttribut ;           // Nom de l'attribut que représente le texte
    private String  nomAttributIncrBloc ;   // Nom de l'attribut indiquant la taille des blocs (quand on clique à côté du curseur)
    private boolean valOpposee ;            // Indique si la valeur de la barre représente la valeur opposée à celle de l'attribut (par exemple les latitude : les latitudes augmentent quand la barre monte)
    private int     multiplicateur ;        // Multiplicateur entre la valeur représentée (telle qu'elle se présente dans l'objet observé) et la valeur à utiliser dans la barre (qui peut être graduée plus finement)
    private String  modeleInfobulle ;       // Texte affiché comme infobulle. Des éléments sont remplacés lors de l'affichage.
    
    
    
    /**
     * Constructeur.
     * @param objetObserve        Objet observé.
     * @param nomAttribut         Nom de l'attribut observé.
     * @param nomAttributIncrBloc Nom de l'attribut indiquant la taille de l'incrément par bloc.
     * @param modeleInfobulle     Modèle de texte pour l'infobulle.
     * @param orientation         Orientation de la barre.
     * @param valCourante         Valeur à donner à l'attribut.
     * @param etendue             Etendue couverte par le curseur.
     * @param valMin              Valeur minimale.
     * @param valMax              Valeur maximale.
     * @param multiplicateur      Multiplicateur entre la valeur de l'attribut et celle du composant.
     */
    public BarreDefilement (Observable objetObserve, String nomAttribut, String nomAttributIncrBloc,
                            String modeleInfobulle,
                            int orientation, int valCourante, int etendue, int valMin, int valMax,
                            int multiplicateur)
    {
        this (objetObserve, nomAttribut, nomAttributIncrBloc, modeleInfobulle,
              orientation, valCourante, etendue, valMin, valMax, false, multiplicateur) ;
    }
    /**
     * Constructeur.
     * @param objetObserve        Objet observé.
     * @param nomAttribut         Nom de l'attribut observé.
     * @param nomAttributIncrBloc Nom de l'attribut indiquant la taille de l'incrément par bloc.
     * @param modeleInfobulle     Modèle de texte pour l'infobulle.
     * @param orientation         Orientation de la barre.
     * @param valCourante         Valeur à donner à l'attribut.
     * @param etendue             Etendue couverte par le curseur.
     * @param valMin              Valeur minimale.
     * @param valMax              Valeur maximale.
     * @param valOpposee          Indique si la valeur de l'attribut et celle du composant sont de signe contraire.
     * @param multiplicateur      Multiplicateur entre la valeur de l'attribut et celle du composant.
     */
    public BarreDefilement (Observable objetObserve, String nomAttribut, String nomAttributIncrBloc,
                            String modeleInfobulle,
                            int orientation, int valCourante, int etendue, int valMin, int valMax,
                            boolean valOpposee, int multiplicateur)
    {
        super (orientation, valCourante * multiplicateur, etendue * multiplicateur, valMin * multiplicateur, valMax * multiplicateur) ;
        this.setUnitIncrement (multiplicateur) ;
        
        // Inscrire l'objet comme observateur
        objetObserve.addObserver (this) ;
        
        // Mémoriser les paramètres
        this.objetObserve        = objetObserve ;
        this.nomAttribut         = nomAttribut ;
        this.nomAttributIncrBloc = nomAttributIncrBloc ;
        this.valOpposee          = valOpposee ;
        this.multiplicateur      = multiplicateur ;
        this.modeleInfobulle     = modeleInfobulle ;
        
        // Représenter l'élément
        ajusterRepresentation() ;
    }
    
    
    /**
     * Réception d'une notification de modification de l'état de l'application.
     */
    public void update (Observable observe, Object param)
    {
        String eltModifie = (String) param ;
        
        // Ajuster la représentation du composant s'il est concerné
        if (this.nomAttribut.equals (eltModifie) || this.nomAttributIncrBloc.equals (eltModifie))
            ajusterRepresentation() ;
    }
    
    
    /**
     * Ajuste le composant en fonction de la valeur de l'attribut qui lui est associé.
     */
    public void ajusterRepresentation ()
    {
        // Récupérer la valeur des attributs
        // (valeur représentée par la barre)
        Object valAttribut = Objets.valAttribut (this.objetObserve, this.nomAttribut) ;
        int valEntiere = (int) ((valAttribut instanceof Integer ?
                                   (double)(Integer) valAttribut :
                                   ((Double)valAttribut))
                               * this.multiplicateur) ;
        if (this.valOpposee)
            valEntiere = -valEntiere ;
        // (valeur de l'incrément par bloc)
        Object valAttributIncrBloc = Objets.valAttribut (this.objetObserve, this.nomAttributIncrBloc) ;
        int valIncrBloc = (int) ((valAttributIncrBloc instanceof Integer ?
                                    (double)(Integer) valAttributIncrBloc :
                                    ((Double)valAttributIncrBloc))
                                * this.multiplicateur) ;

        // Ajuster la représentation du composant
        this.setValue          (valEntiere) ;
        this.setToolTipText    (this.modeleInfobulle.replace (MARQUEUR_REMPL_VAL_ENTIER, ""+valAttribut)) ;
        this.setBlockIncrement (valIncrBloc) ;
    }
    

    
    // Accesseurs
    public Object objetObserve    () { return this.objetObserve ; }
    public String nomAttribut     () { return this.nomAttribut ; }
    public String modeleInfobulle () { return this.modeleInfobulle ; }
    public Object valeur          () { return this.getValue() ; }

}
