/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique.actions;

import static fr.histoiremondiale.histoire.utiles.es.fichiers.Fichiers.urlLocaleDeNomFichier;

import java.awt.Component;
import java.io.File;

import javax.swing.JOptionPane;

import fr.histoiremondiale.histoire.DonneesIGraphique;
import fr.histoiremondiale.histoire.EtatAppli;
import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.donnees.Territoire;
import fr.histoiremondiale.histoire.igraphique.DialAPropos;
import fr.histoiremondiale.histoire.igraphique.DialLimiteVisibiliteTerritoire;
import fr.histoiremondiale.histoire.igraphique.FenPageHTML;
import fr.histoiremondiale.histoire.igraphique.FenPrincipale;



/**
 * Actions sur les fenêtres et boîtes de dialogue de l'application.
 */
public class ActionsFenetres
{

    /**
     * Affiche la boîte de dialogue permettant de définir le niveau de visibilité d'un territoire.
     * @param territoire Le territoire concerné.
     */
    public static void affDialVisibiliteTerritoire (Territoire territoire)
    {
        HistoireMondiale  application       = HistoireMondiale.instance() ;
        FenPrincipale     fenPrincipale     = application.fenPrincipale() ;
        DonneesIGraphique donneesIGraphique = application.donneesIGraphique() ;
        
        // Afficher la boîte de dialogue
        double mini   = EtatAppli.MINI_LOUPE ;
        double maxi   = EtatAppli.MAXI_LOUPE ;
        double valeur = donneesIGraphique.loupeTerritoire (territoire) ;
        double loupe  = application.etat().loupe() ;
        
        DialLimiteVisibiliteTerritoire dial = new DialLimiteVisibiliteTerritoire (fenPrincipale, territoire, mini, maxi, valeur, loupe) ;
        dial.positionner () ;
        dial.setVisible (true) ;
    }
    
    
    /**
     * Affiche la boîte de dialogue de choix d'une civilisation.
     */
    public static void affDialChoixCivilisation ()
    {
        HistoireMondiale.instance().dialCivilisations().setVisible(true) ;        
    }
    
    
    /**
     * Affiche la boîte de dialogue de choix d'une date.
     */
    public static void affDialChoixDate ()
    {
        HistoireMondiale application   = HistoireMondiale.instance() ;
        EtatAppli        etatAppli     = application.etat() ;
        int              anneeActuelle = etatAppli.annee() ;
        application.dialAnnee().showChoixAnnee (anneeActuelle) ;
    }
    
    
    /**
     * Masque la boîte de dialogue permettant de choisir une date.
     */
    public static void masquerDialChoixAnnee ()
    {
        HistoireMondiale.instance().dialAnnee().setVisible (false) ;
    }
    
    
    /**
     * Affiche la fenêtre de présentation de l'application (aide en ligne).
     */
    public static void affFenPresentation ()
    {
        String chemDocumentation = HistoireMondiale.instance().config().chemRepDocumentation() ;
        String urlPage = urlLocaleDeNomFichier (chemDocumentation + File.separator + "presentationjava.html") ;
        // Afficher la page choisie
        try
        {
            FenPageHTML fenHtml = new FenPageHTML ("Présentation du logiciel Histoire", urlPage, 1500, 600) ;
            fenHtml.setVisible (true) ;
        }
        catch (Throwable e)
        {
            e.printStackTrace() ;
            JOptionPane.showMessageDialog (null, "Erreur lors de l'affichage de la présentation.", "Erreur d'affichage de la page", JOptionPane.ERROR_MESSAGE) ;
        }
    }
    
    
    /**
     * Affiche la boîte de dialogue "A propos".
     */
    public static void affDialAPropos ()
    {
        new DialAPropos(HistoireMondiale.instance().fenPrincipale()).setVisible (true) ;
    }

    
    /**
     * Affiche la boîte de dialogue permettant de choisir le niveau de loupe.
     */
    public static void affDialLoupe ()
    {
    	HistoireMondiale.instance().dialLoupe().afficher(HistoireMondiale.instance().etat().loupe()) ; 
    }
    
    
    /**
     * Masque la boîte de dialogue permettant de choisir le niveau de loupe.
     */
    public static void masquerDialLoupe ()
    {
        HistoireMondiale.instance().dialLoupe().setVisible (false) ;
    }
    
    
    /**
     * Affiche le menu contextuel de réglage des déplacements dans l'espace de la fenêtre principale.
     * @param composant Le composant sur lequel faire apparaître le menu.
     * @param x         Abscisse du point auquel faire apparaître le menu sur le composant.
     * @param y         Ordonnée du point auquel faire apparaître le menu sur le composant.
     */
    public static void affMenuContextuelDeplEspace (Component composant, int x, int y)
    {
        HistoireMondiale.instance().fenPrincipale().menuctx_deplEspace.show (composant, x, y) ;
    }
    
    /**
     * Affiche le menu contextuel de réglage des déplacements dans le temps de la fenêtre principale.
     * @param composant Le composant sur lequel faire apparaître le menu.
     * @param x         Abscisse du point auquel faire apparaître le menu sur le composant.
     * @param y         Ordonnée du point auquel faire apparaître le menu sur le composant.
     */
    public static void affMenuContextuelDeplAnnees (Component composant, int x, int y)
    {
        HistoireMondiale.instance().fenPrincipale().menuctx_deplAnnees.show (composant, x, y) ;
    }


	public static void affDialCentre() 
	{
		HistoireMondiale.instance().dialCentre().afficher(HistoireMondiale.instance().etat().ptCentralCarte()) ; 		
	}

	public static void masquerDialCentre() 
	{
        HistoireMondiale.instance().dialCentre().setVisible (false) ;
	}
	
}
