/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique;

import static fr.histoiremondiale.histoire.EtatAppli.PRECISION_ESPACE;
import static fr.histoiremondiale.histoire.EtatAppli.VALS_PAS_ANNEES;
import static fr.histoiremondiale.histoire.EtatAppli.VALS_PAS_ESPACE;
import static fr.histoiremondiale.histoire.RessourcesAppli.ICONE_APPLI_32;
import static fr.histoiremondiale.histoire.RessourcesAppli.ICONE_LOUPE;
import static java.lang.Math.pow ;

import java.awt.BorderLayout;
import java.awt.Color ;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.KeyEvent;
import java.awt.event.MouseWheelEvent ;
import java.awt.event.MouseWheelListener ;
import java.beans.PropertyChangeEvent ;
import java.beans.PropertyChangeListener ;
import java.util.Observable ;
import java.util.Observer ;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup ;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent ;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem ;
import javax.swing.JPanel;
import javax.swing.JPopupMenu ;
import javax.swing.JScrollBar;
import javax.swing.JSplitPane;
import javax.swing.border.BevelBorder ;

import fr.histoiremondiale.histoire.EtatAppli;
import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.RessourcesAppli;
import fr.histoiremondiale.histoire.donnees.Territoire;
import fr.histoiremondiale.histoire.donnees.TexteTerritoire;
import fr.histoiremondiale.histoire.igraphique.composants.BarreDefilement;
import fr.histoiremondiale.histoire.igraphique.composants.EltMenuCoche;
import fr.histoiremondiale.histoire.igraphique.composants.GlissiereDates;
import fr.histoiremondiale.histoire.igraphique.composants.Texte;
import fr.histoiremondiale.histoire.utiles.math.PointSphere;



/**
 * Fenêtre principale de l'application.
 */
public class FenPrincipale extends JFrame implements Observer, MouseWheelListener
{

    // Constantes
    public static final double PROPORTION_SEPARATEUR_DEFAUT = 0.66 ;                                // Proportion de l'espace du panneau à séparateur alloué par défaut au composant de gauche
    public static final int    MULTIPLICATEUR_ESPACE        = (int) pow (10, PRECISION_ESPACE) ;    // Multiplicateur entre l'unité de défilement dans l'espace (le degré) et le nombre de crans dans les barres de défilement ("graduées" en dessous du degré)

    
    // Menus
    public JMenu menu_fichier ;
    public JMenuItem eltmenu_ficReinitValeurs ;
    public JMenuItem eltmenu_ficExporterCarte ;
    public JMenuItem eltmenu_ficExporterCarteEtParagraphe ;
    public JMenuItem eltmenu_ficVisualiserParagraphe ;
    public JMenuItem eltmenu_ficQuitter ;
    public JMenu menu_navigation ;
    public JMenuItem eltmenu_navigChoisirCivilisation ;
    public JMenuItem eltmenu_navigChoisirDate ;
    public JMenuItem eltmenu_navigChoisirCentre ;
    public JMenu menu_affichage ;
    public JMenuItem eltmenu_affFleuvesVisibles ;
    public JMenuItem eltmenu_affChoixLoupe ;
    public JMenu menu_options ;
    public JMenu menu_deplacements ;
    public JMenu menu_deplAnnees ;
    public JMenu menu_deplEspace ;
    public JMenu menu_parametres ;
    public JMenuItem eltmenu_optInfobullesCarte ;
    public JMenuItem eltmenu_optSymboleTracer ;
    public JMenuItem eltmenu_optTexteSurvolFondClair ;
    public JMenuItem eltmenu_affMeridiensParallelesVisibles;
    public JMenuItem eltmenu_optSimpleClicFermerParagraphes ;
    public JMenuItem eltmenu_optMoletteGlisseDeUn ;
    public JMenuItem eltmenu_optAffParagraphesHtmlGeneres ;
    public JMenu menu_aide ;
    public JMenuItem eltmenu_aidePresentation ;
    public JMenuItem eltmenu_aideAPropos ;
    
    // Menus contextuels
    public JPopupMenu menuctx_deplAnnees ;
    public JPopupMenu menuctx_deplEspace ;
    
    // Panneaux principaux
    private JSplitPane    sp_panneauDeuxCotes ;     // Panneau séparant le navigateur et la carte
    public  PanNavigateur p_navigateur ;            // Panneau contenant le navigateur html
    public  PanCarte      p_carte ;                 // Panneau sur lequel est tracée la carte
    
    // Autres composants
    private Box     p_infos ;                       // Panneau d'informations
    public  Texte   txt_pointCentral ;              // Informations sur l'état de la carte
    public  Texte   txt_annee ;
    public  JLabel  txt_loupe ;
    public  JLabel  txt_infosTerritoireSurvole ;    // Informations sur le territoire survolé
    public  JLabel  txt_infosRoiTerritoireSurvole ;
    public  JButton bt_loupe ;                      // Loupe
    
    public BarreDefilement defil_longitudes ;       // Barres de défilement (date et position géographique)
    public BarreDefilement defil_latitudes ;
    
    public GlissiereDates  defil_annees ;
    


    /**
     * Constructeur.
     */
    public FenPrincipale ()
    {
        super ("Histoire") ;
        
        System.out.println("\t\tDébut de la construction de la fenêtre");
        long debFen = System.currentTimeMillis() ;
        

        HistoireMondiale application = HistoireMondiale.instance() ;
        EtatAppli        etatAppli   = application.etat() ;
        
        
        // Icône
        this.setIconImage (RessourcesAppli.imageIcone (ICONE_APPLI_32)) ;

        
        // Informations
        // (créer les objets graphiques)
        this.txt_pointCentral      = new Texte (" Centre : " + Texte.MARQUEUR_REMPL_VAL_CHAINE   + " ", etatAppli, "ptCentralCarte") ;
        this.txt_annee             = new Texte (" Date : "   + Texte.MARQUEUR_REMPL_VAL_CHAINE   + " ", etatAppli, "annee") ;
        this.txt_loupe             = new Texte (" Loupe : "  + Texte.MARQUEUR_REMPL_VAL_FLOTTANT + "x", etatAppli, "loupe") ;
        this.txt_infosTerritoireSurvole    = new JLabel() ;
        this.txt_infosRoiTerritoireSurvole = new JLabel() ;
        this.txt_infosTerritoireSurvole.setOpaque     (true) ;
        this.txt_infosRoiTerritoireSurvole.setOpaque  (true) ;
        this.txt_infosTerritoireSurvole.setVisible    (false) ;
        this.txt_infosRoiTerritoireSurvole.setVisible (false) ;
        this.txt_pointCentral.setBorder        (new BevelBorder (BevelBorder.LOWERED)) ;
        this.txt_annee.setBorder               (new BevelBorder (BevelBorder.LOWERED)) ;
        this.txt_loupe.setBorder               (new BevelBorder (BevelBorder.LOWERED)) ;
        // (assembler le panneau)
        this.p_infos = new Box (BoxLayout.LINE_AXIS) ;
        this.p_infos.add (this.txt_pointCentral) ;
        this.p_infos.add (Box.createHorizontalStrut (5)) ;
        this.p_infos.add (this.txt_annee) ;
        this.p_infos.add (Box.createHorizontalStrut (5)) ;
        this.p_infos.add (this.txt_loupe) ;
        this.p_infos.add (Box.createHorizontalGlue()) ;
        this.p_infos.add (this.txt_infosTerritoireSurvole) ;
        this.p_infos.add (this.txt_infosRoiTerritoireSurvole) ;
        
        // Navigation dans l'espace
        PointSphere ptCentralCarte = etatAppli.ptCentralCarte() ;
        final int angle = PanCarte.ANGLE_CARTE ;    

        this.defil_longitudes = new BarreDefilement (etatAppli, "longitude", "deplEspace",
                                                     BarreDefilement.MARQUEUR_REMPL_VAL_ENTIER + "°",
                                                     JScrollBar.HORIZONTAL, (int)-ptCentralCarte.longitude(),
                                                     angle, -179 - 1, +180 + 1 + angle, MULTIPLICATEUR_ESPACE) ;
        this.defil_longitudes.addMouseWheelListener (this) ;
        this.defil_latitudes = new BarreDefilement (etatAppli, "latitude", "deplEspace",
                                                    BarreDefilement.MARQUEUR_REMPL_VAL_ENTIER + "°",
                                                    JScrollBar.VERTICAL, (int)-ptCentralCarte.latitude(),
                                                    angle, -90, +90 + angle, true, MULTIPLICATEUR_ESPACE) ;
        this.defil_latitudes.addMouseWheelListener (this) ;
        
        // Loupe
        int tailleBtLoupe = (int) this.defil_longitudes.getPreferredSize().getHeight() - 2 ;
        this.bt_loupe = new JButton (RessourcesAppli.icone (ICONE_LOUPE)) ;
        this.bt_loupe.setToolTipText   ("Clic gauche pour augmenter, clic droit pour diminuer") ;
        this.bt_loupe.setPreferredSize (new Dimension (tailleBtLoupe, tailleBtLoupe)) ;
        this.bt_loupe.setMaximumSize   (new Dimension (tailleBtLoupe, tailleBtLoupe)) ;

        // Navigation dans le temps
        int valCourante = etatAppli.annee();
        int valMin      = -3100;
        int valMax      = 2000;
        int deplacement = VALS_PAS_ANNEES[HistoireMondiale.instance().etat().iDeplAnnees()] ;
        
        this.defil_annees = new GlissiereDates (etatAppli, "annee", "deplAnnees", valMin, valMax, valCourante, deplacement) ;
        this.defil_annees.addMouseWheelListener (this) ;
        
        // (panneau contenant les éléments de défilement des années)
        Box p_defilAnnees = new Box (BoxLayout.X_AXIS) ;
        p_defilAnnees.add (this.defil_annees) ;

        // Panneau regroupant la carte et le navigateur
        // (la construction de cette partie est terminée plus tard)
        this.sp_panneauDeuxCotes = new JSplitPane (JSplitPane.HORIZONTAL_SPLIT, false, new JPanel(), new JPanel()) ;
        this.sp_panneauDeuxCotes.setOneTouchExpandable (true) ;
        // this.sp_panneauDeuxCotes.setResizeWeight (PROPORTION_SEPARATEUR_DEFAUT) ;
        this.sp_panneauDeuxCotes.setResizeWeight (etatAppli.posSeparateurCarteNavig()) ;
        // (maintenir la proportion entre les deux panneaux, carte et navigateur, lors des redimensionnements)
        this.sp_panneauDeuxCotes.addPropertyChangeListener(JSplitPane.DIVIDER_LOCATION_PROPERTY, 
                new PropertyChangeListener()
                {
                    public void propertyChange(PropertyChangeEvent pce)
                    {
                        sp_panneauDeuxCotes.setResizeWeight (posSeparateurPanneaux()) ;
                    }
                });

        // Assembler la partie basse de la fenêtre
        Box p_sud = new Box (BoxLayout.Y_AXIS) ;
        p_sud.add (p_defilAnnees) ;
        p_sud.add (this.p_infos) ;

        // Configuration et affichage de la fenêtre
        this.setDefaultCloseOperation (JFrame.DO_NOTHING_ON_CLOSE) ;    // (fermeture de l'appli gérée à la main)
        //this.addWindowListener      (this) ;
        // (calculer la taille initiale de la fenêtre [taille à laquelle elle sera réduite si elle n'est plus maximisée])
        Dimension tailleEcran = Toolkit.getDefaultToolkit().getScreenSize() ;
        double largeurFen ;
        double hauteurFen ;
        if (tailleEcran.getWidth() / tailleEcran.getHeight() >= 1.33)
        {
            hauteurFen = 0.75 * tailleEcran.getHeight() ;
            largeurFen = 1.33 * hauteurFen ;
        }
        else
        {
            largeurFen = 0.75 * tailleEcran.getWidth() ;
            hauteurFen = largeurFen / 1.33 ;
        }
        this.setSize        (new Dimension ((int)largeurFen, (int)hauteurFen)) ;
        this.setMinimumSize (new Dimension (250, 150)) ;
        // (centrer la fenêtre)
        int xFen = (int) (tailleEcran.getWidth()  - largeurFen) / 2 ;
        int yFen = (int) (tailleEcran.getHeight() - hauteurFen) / 2 ;
        this.setLocation (xFen, yFen) ;

        // Menus
        // (vers la fin, les autres données doivent avoir été initialisées)
        JMenuBar menus = this.creerMenus() ;

        // Assembler la fenêtre
        this.setJMenuBar (menus) ;
        this.getContentPane().setLayout (new BorderLayout()) ;
        this.getContentPane().add (this.sp_panneauDeuxCotes, BorderLayout.CENTER) ;
        this.getContentPane().add (p_sud,                    BorderLayout.SOUTH) ;
        
        // Mettre à jour certaines informations
        majInfosTerritoireSurvole() ;
        
        // S'inscrire comme observateur de l'état de l'application
        etatAppli.addObserver (this) ;
        
        // Afficher la fenêtre
        this.setExtendedState (JFrame.MAXIMIZED_BOTH) ;

        long finFen = System.currentTimeMillis() ;
        System.out.println ("\t\tFin de la construction de la fenêtre en : " + (finFen - debFen) + "ms.") ;
    }
    
    
    /**
     * Termine la création de la fenêtre.<br>
     * Note : la création de certain composants doit attendre que la fenêtre soit disponible. La fenêtre doit
     *   donc être construite en deux étape : l'appel au constructeur puis cette méthode.<br>
     *      ! Le problème vient surtout de la construction des composants graphique qui se mêle à l'accès au
     *        gestionnaire d'événements : le gestionnaire d'événements doit être prêt quand on construit le
     *        panneau des navigateurs html, parce que cette construction lance la construction des composants
     *        représentants les paragraphes, auxquels on associe des événements au fur et à mesure. Et la
     *        fenêtre doit être créée pour pouvoir créer le gestionnaire de composants.
     */
    public void finirInitialisation ()
    {
    	System.out.println("début de la fin de l'initialisation");
    	long debut = System.currentTimeMillis();

    	HistoireMondiale application = HistoireMondiale.instance() ;
        EtatAppli        etatAppli   = application.etat() ;
        

        long dateDeb, dateFin;
        dateDeb = System.currentTimeMillis();
        // Créer le panneau pour le navigateur html
        this.p_navigateur = new PanNavigateur() ;
        dateFin = System.currentTimeMillis();
        System.out.println("\tFin de la construction du panneau navigateur en : " + (dateFin - dateDeb) + " ms."); 
        
        dateDeb = System.currentTimeMillis();
        // Créer le panneau de la carte
        this.p_carte = new PanCarte() ;
        
        // Assembler la carte navigable
        JPanel p_carteNavigable = new JPanel (new BorderLayout()) ;
        Box p_basCarte = new Box (BoxLayout.LINE_AXIS) ;
        p_basCarte.add (this.defil_longitudes) ;
        p_basCarte.add (this.bt_loupe) ;
        p_carteNavigable.add (this.p_carte,         BorderLayout.CENTER) ;
        p_carteNavigable.add (p_basCarte,           BorderLayout.SOUTH) ;
        p_carteNavigable.add (this.defil_latitudes, BorderLayout.EAST) ;
        dateFin = System.currentTimeMillis();
        System.out.println("\tFin de la construction du panneau carte en : " + (dateFin - dateDeb) + " ms."); 
        
        // Panneau regroupant la carte et le navigateur
        // Note : Avant que des événements n'y touchent, ça fonctionne si on le met ici mais pas au début de la méthode !)
        //          => Des problèmes d'interaction avec Swing, parce qu'on n'exécute pas cette méthode dans le
        //             fil d'exécution de Swing (et c'est mal) ? A voir si on modifie cette séquence de création
        //             plus tard...
        // Placer les composants dans le panneau avec séparateur
        this.sp_panneauDeuxCotes.setLeftComponent   (p_carteNavigable) ;
        this.sp_panneauDeuxCotes.setRightComponent  (this.p_navigateur) ;
        this.sp_panneauDeuxCotes.setDividerLocation (etatAppli.posSeparateurCarteNavig()) ;
        
        dateFin = System.currentTimeMillis();
        System.out.println("Fin de l'assemblage de la fenêtre principale en : " + (dateFin-debut) + " ms.");
    }
    
    
    // Initialisation des menus
    private JMenuBar creerMenus ()
    {
        EtatAppli etatAppli = HistoireMondiale.instance().etat() ;      // Etat de l'appli
        JMenuBar  menus     = new JMenuBar() ;                          // Barre de menus à renvoyer
        
        
        // Créer les menus
        menus.add (this.menu_fichier = menu ("Fichier", KeyEvent.VK_F)) ;
            this.menu_fichier.add (this.eltmenu_ficReinitValeurs             = new JMenuItem ("Revenir aux valeurs initiales")) ;
            this.menu_fichier.addSeparator() ;
            this.menu_fichier.add (this.eltmenu_ficExporterCarte             = new JMenuItem ("Sauvegarder la carte dans...")) ;
            this.menu_fichier.add (this.eltmenu_ficExporterCarteEtParagraphe = new JMenuItem ("Sauvegarder le paragraphe dans...")) ;
            this.menu_fichier.add (this.eltmenu_ficVisualiserParagraphe      = new JMenuItem ("Visualiser un paragraphe sauvegardé...")) ;
            this.menu_fichier.addSeparator() ;
            this.menu_fichier.add (this.eltmenu_ficQuitter                   = new JMenuItem ("Quitter")) ;
        menus.add (this.menu_navigation = menu ("Navigation", KeyEvent.VK_N)) ;
            this.menu_navigation.add (this.eltmenu_navigChoisirCivilisation  = new JMenuItem ("Choisir une civilisation...")) ;
            this.menu_navigation.add (this.eltmenu_navigChoisirDate          = new JMenuItem ("Choisir une date...")) ;
            this.menu_navigation.add (this.eltmenu_navigChoisirCentre        = new JMenuItem ("Choisir une position...")) ;
        menus.add (this.menu_affichage = menu ("Affichage", KeyEvent.VK_A)) ;
            this.menu_affichage.add (this.eltmenu_affFleuvesVisibles             = new JCheckBoxMenuItem ("Fleuves visibles",                etatAppli.fleuvesAffiches())) ;
            this.menu_affichage.add (this.eltmenu_affMeridiensParallelesVisibles = new JCheckBoxMenuItem ("Méridiens & parallèles visibles", etatAppli.meridiensParallelesAffiches()));
            this.menu_affichage.add (this.eltmenu_affChoixLoupe                  = new JMenuItem         ("Choisir la loupe...")) ;
        menus.add (this.menu_options = menu ("Options", KeyEvent.VK_O)) ;
            this.menu_options.add (menus.add (this.menu_deplacements = menu ("Déplacements", null))) ;
                this.menu_deplacements.add (this.menu_deplAnnees = menuDeplAnnees()) ;
                this.menu_deplacements.add (this.menu_deplEspace = menuDeplEspace()) ;
            this.menu_options.add (menus.add (this.menu_parametres = menu ("Paramètres", null))) ;
                this.menu_parametres.add (this.eltmenu_optMoletteGlisseDeUn           = new JCheckBoxMenuItem ("La molette déplace les glissières de 1 seulement",          etatAppli.moletteGlisseDeUn())) ;
                this.menu_parametres.add (this.eltmenu_optSymboleTracer               = new JCheckBoxMenuItem ("Afficher le symbole de tracer en cours",                    etatAppli.symboleAttenteTracer())) ;
                this.menu_parametres.add (this.eltmenu_optInfobullesCarte             = new JCheckBoxMenuItem ("Infobulles carte",                                          etatAppli.infobullesCarte())) ;
                this.menu_parametres.add (this.eltmenu_optTexteSurvolFondClair        = new JCheckBoxMenuItem ("Texte du territoire survolé sur fond clair",                etatAppli.texteSurvolFondClair())) ;
                this.menu_parametres.add (this.eltmenu_optSimpleClicFermerParagraphes = new JCheckBoxMenuItem ("Un simple clic ferme la boîte de dialogue des paragraphes", etatAppli.simpleClicFermerParagraphes())) ;
                this.menu_parametres.add (this.eltmenu_optAffParagraphesHtmlGeneres   = new JCheckBoxMenuItem ("Afficher le paragraphe html sauvegardé pour vérification",  etatAppli.affParagraphesHtmlGeneres())) ;
        // ("?" ça aurait été bien comme nom de menu, mais il n'y a pas de touche portant ce nom
        //  http://docs.oracle.com/javase/1.4.2/docs/api/java/awt/event/KeyEvent.html)
        menus.add (this.menu_aide = menu ("Aide", KeyEvent.VK_E)) ;
            this.menu_aide.add (this.eltmenu_aidePresentation = new JMenuItem ("Présentation")) ;
            this.menu_aide.add (this.eltmenu_aideAPropos      = new JMenuItem ("A propos")) ;

        // Créer les menus contextuels
        this.menuctx_deplAnnees = menuContextuelDeplAnnees() ;
        this.menuctx_deplEspace = menuContextuelDeplEspace() ;
            
        // Renvoyer la barre de menus
        return menus ;
    }
    
    
    // Crée un menu
    // Touche de raccourci optionnelle : KeyEvent.VK_...
    private JMenu menu (String nom, Integer toucheRaccourci)
    {
        JMenu menu = new JMenu (nom) ;
        if (toucheRaccourci != null)
            menu.setMnemonic (toucheRaccourci) ;
        return menu ;
    }

    
    // Crée le menu de sélection du pas des dates
    private JMenu menuDeplAnnees ()
    {
        JMenu menu = new JMenu ("Date") ;
        remplirMenuDeplAnnees (menu) ;
        return menu ;
    }
    private JPopupMenu menuContextuelDeplAnnees ()
    {
        // (ne fonctionne pas : l'affichage du menu ainsi récupéré supprime le menu "normal" de l'endroit
        //  où il se trouvait avant (comme sous-menu d'un menu de la barre))
        // return menuDeplEnnees().getPopupMenu() ;
        JPopupMenu menu = new JPopupMenu() ;
        remplirMenuDeplAnnees (menu) ;
        return menu ;
    }
    // (JMenu et JPopupMenu n'ont pas de classe mère ou d'interface en commun)
    private void remplirMenuDeplAnnees (JComponent menu)
    {
        try
        {
            ButtonGroup groupeCoches = new ButtonGroup() ;
    
            // Créer les éléments de menu
            for (int i = 0 ; i < VALS_PAS_ANNEES.length ; i++)
            {
                int pasDate = VALS_PAS_ANNEES[i];
                String texte = "" + pasDate + " an" + (pasDate == 1 ? "" : "s") ;
                EltMenuCoche eltMenu = new EltMenuCoche (texte, HistoireMondiale.instance().etat(), "iDeplAnnees", i) ;
                menu.add         (eltMenu) ;
                groupeCoches.add (eltMenu) ;
            }
        }
        catch (Exception e)
        {
            throw new IllegalArgumentException ("Erreur lors de la création du menu déplacement années",e) ;
        }
    }


    
    // Crée le menu de sélection du pas des coordonnées
    private JMenu menuDeplEspace ()
    {
        JMenu menu = new JMenu ("Espace") ;
        remplirMenuDeplEspace (menu) ;
        return menu ;
    }
    private JPopupMenu menuContextuelDeplEspace ()
    {
        // (ne fonctionne pas : l'affichage du menu ainsi récupéré supprime le menu "normal" de l'endroit
        //  où il se trouvait avant (comme sous-menu d'un menu de la barre))
        // return menuDeplEspace().getPopupMenu() ;
        JPopupMenu menu = new JPopupMenu() ;
        remplirMenuDeplEspace (menu) ;
        return menu ;
    }
    // (JMenu et JPopupMenu n'ont pas de classe mère ou d'interface en commun)
    private void remplirMenuDeplEspace (JComponent menu)
    {
        ButtonGroup groupeCoches = new ButtonGroup() ;

        // Créer les éléments de menu
        for (int i = 0 ; i < VALS_PAS_ESPACE.length ; i++)
        {
            String texte = "" + VALS_PAS_ESPACE[i] + "°" ;
            EltMenuCoche eltMenu = new EltMenuCoche (texte, HistoireMondiale.instance().etat(), "iDeplEspace", i) ;
            menu.add         (eltMenu) ;
            groupeCoches.add (eltMenu) ;
        }
    }
    
    
    
    /**
     * Renvoie la position du séparateur entre les panneaux (carte et navigateur).
     * @return La position du séparateur entre les panneaux.
     */
    public double posSeparateurPanneaux ()
    {
        return (double) this.sp_panneauDeuxCotes.getDividerLocation()
                      / (this.sp_panneauDeuxCotes.getWidth() - this.sp_panneauDeuxCotes.getDividerSize()) ; // (inspiré du code de JSplitPane.setDividerLocation (double) )
    }
    
    
    
    /**
     * Met à jour les informations concernant le territoire survolé.
     */
    public void majInfosTerritoireSurvole ()
    {
        EtatAppli  etatAppli         = HistoireMondiale.instance().etat() ;
        Territoire territoireSurvole = etatAppli.territoireSurvole() ;
        
        
        // Pas de territoire : infos vides
        if (territoireSurvole == null)
        {
            this.txt_infosTerritoireSurvole.setText       ("") ;
            this.txt_infosRoiTerritoireSurvole.setText    ("") ;
            this.txt_infosTerritoireSurvole.setVisible    (false) ;
            this.txt_infosRoiTerritoireSurvole.setVisible (false) ;
        }
        else
        {
            // Récupérer les informations sur le territoire
            int             annee         = etatAppli.annee() ;
            String          nomTerritoire = territoireSurvole.nom().replace ("$", " ") ;
            TexteTerritoire txtTerritoire = territoireSurvole.texteEn (annee) ;
            
            // Créer les textes à afficher
            String txtTerr  = nomTerritoire ;
            String txtCompl = (txtTerritoire == null ? "" : txtTerritoire.texte().replace ("&", " ")) ;
            
            // Sélectionner les couleurs du texte et du fond
            if (etatAppli.texteSurvolFondClair())
            {
                this.txt_infosTerritoireSurvole.setForeground (Color.BLACK) ;
                this.txt_infosTerritoireSurvole.setBackground (this.p_infos.getBackground()) ;
                this.txt_infosRoiTerritoireSurvole.setForeground (Color.BLACK) ;
                this.txt_infosRoiTerritoireSurvole.setBackground (this.p_infos.getBackground()) ;
                this.txt_infosTerritoireSurvole.setBorder    (new BevelBorder (BevelBorder.LOWERED)) ;
                this.txt_infosRoiTerritoireSurvole.setBorder (new BevelBorder (BevelBorder.LOWERED)) ;

            }
            else
            {
                this.txt_infosTerritoireSurvole.setForeground (Color.WHITE) ;
                this.txt_infosTerritoireSurvole.setBackground (Color.BLACK) ;
                this.txt_infosRoiTerritoireSurvole.setForeground (Color.WHITE) ;
                this.txt_infosRoiTerritoireSurvole.setBackground (Color.BLACK) ;
                this.txt_infosTerritoireSurvole.setBorder    (new BevelBorder (BevelBorder.LOWERED, Color.BLACK, Color.GRAY)) ;
                this.txt_infosRoiTerritoireSurvole.setBorder (new BevelBorder (BevelBorder.LOWERED, Color.BLACK, Color.GRAY)) ;
            }
            
            // Afficher les textes
            this.txt_infosTerritoireSurvole.setText       (" " + txtTerr + " ") ;
            this.txt_infosRoiTerritoireSurvole.setText    (" " + txtCompl  + " ") ;
            this.txt_infosTerritoireSurvole.setVisible    (true) ;
            this.txt_infosRoiTerritoireSurvole.setVisible (! txtCompl.trim().isEmpty()) ;
        }
        
    }

    
    /**
     * Méthode de traitement des modifications de l'état de l'appli.
     */
    public void update (Observable o, Object params)
    {
        String nomAttribut = (String) params ;
        
        // Si le territoire survolé change, mettre à jour les informations
        if ("territoireSurvole".equals (nomAttribut))
            majInfosTerritoireSurvole() ;
    }



    /**
     * Gestion de la molette sur certains composants (interactions internes au panneau).
     */
    // Pour info, solution plus "Swing" (donc plus compliquée) :
    //   http://www.developpez.net/forums/d279793/java/interfaces-graphiques-java/awt-swing/jslider-controler-intervalles-deplacement-curseur/
    public void mouseWheelMoved (MouseWheelEvent evt)
    {
        EtatAppli etatAppli = HistoireMondiale.instance().etat() ;
        Object    source    = evt.getSource() ;
        
        // Récupérer la valeur de décalage
        int decalageEspace = (etatAppli.moletteGlisseDeUn() ? 1 : etatAppli.deplEspace()) * MULTIPLICATEUR_ESPACE ;
        int decalageAnnees = (etatAppli.moletteGlisseDeUn() ? 1 : etatAppli.deplAnnees()) ;
        
        // Traiter l'événement
        if (source == this.defil_longitudes)
            this.defil_longitudes.setValue (this.defil_longitudes.getValue() + evt.getWheelRotation() * decalageEspace) ;
        else if (source == this.defil_latitudes)
            this.defil_latitudes.setValue (this.defil_latitudes.getValue() + evt.getWheelRotation() * decalageEspace) ;
        else if (source == this.defil_annees)
            this.defil_annees.setValeur (this.defil_annees.getValeur() + evt.getWheelRotation() * decalageAnnees) ;
        else
            throw new IllegalStateException ("Source inconnue : " + source) ;
    }

}
