/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique;

import static fr.histoiremondiale.histoire.utiles.es.fichiers.Fichiers.urlLocaleDeNomFichier;

import java.awt.BorderLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.IOException;

import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.text.AttributeSet;
import javax.swing.text.Document;
import javax.swing.text.Element;
import javax.swing.text.html.HTML;
import javax.swing.text.html.HTMLDocument;

import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.utiles.exceptions.Exceptions;
import fr.histoiremondiale.histoire.utiles.igraphique.swing.Fenetres;

/**
 * Affiche une page HTML.
 */
public class FenPageHTML extends JFrame implements MouseListener
{

    private String      urlBasePage ;       // URL de base de la page, sans ancre
    private JEditorPane navigHTML ;         // Navigateur HTML de la page
    
    
    
    /**
     * Constructeur.
     * @param titre   Titre de la fenêtre.
     * @param url     URL de la page à afficher.
     * @param largeur Largeur de la fenêtre.
     * @param hauteur Hauteur de la fenêtre.
     * @throws IOException En cas de problème d'accès à la page html.
     */
    public FenPageHTML (String titre, String url, int largeur, int hauteur) throws IOException
    {
       this(titre,url,largeur,hauteur,null);
    }
    
    
    /**
     * Constructeur.
     * @param titre   Titre de la fenêtre.
     * @param url     URL de la page à afficher.
     * @param width Largeur de la fenêtre (de l'intérieur si parent existe).
     * @param heigth Hauteur de la fenêtre (de l'intérieur si parent existe).
     * @param parent  Fenetre contenant notre FenPageHtml, éventuellement null 
     * @throws IOException En cas de problème d'accès à la page html.
     */
    
    public FenPageHTML(String titre, String url, int width, int heigth, JFrame parent) throws IOException
	{
        super(titre);
    	//la largeur et la hauteur
    	int largeur = width, hauteur = heigth;
    	
    	// Mémoriser les paramètres
        this.urlBasePage = url ;
        
        // Construire le navigateur
        this.navigHTML = new JEditorPane() ;
        this.navigHTML.setEditable (false) ;
        this.navigHTML.setPage (url) ;
        this.navigHTML.addMouseListener (this) ;
        
        // Assembler les composants
        this.getContentPane().setLayout (new BorderLayout()) ;
        JScrollPane jsp = new JScrollPane(this.navigHTML);
        this.getContentPane().add (jsp) ;

        if (parent != null)
        {
	   		//une FenPageHtlm est une JFrame contenant un JEditorPane, donc
	   		//avec des marges et un JScrollPane. Si on veut que l'image tienne 
	   		//intégralement (sans JScrolPane horizontal) il faut calculer ces
	   		//diverses marges
	   		//les marges de l'EditorPane
	   		Insets marges = this.navigHTML.getInsets(); 
	       	// +1 pour donner de l'air aux composants
	       	int margesEditorPane = marges.left + marges.right + 1;
	       	int epaisseurScrollBar = new JScrollBar(JScrollBar.VERTICAL).getPreferredSize().width;
	       	
	       	int margesHorizontales = parent.getInsets().left + parent.getInsets().right;
	       	largeur = Math.min(width + margesHorizontales + epaisseurScrollBar + margesEditorPane,
	       									parent.getWidth() - margesHorizontales);
	       	int margesVerticales = parent.getInsets().top + parent.getInsets().bottom + parent.getJMenuBar().getHeight();
	        	hauteur = parent.getHeight()- margesVerticales;
        }
        else parent = HistoireMondiale.instance().fenPrincipale();
        // Paramétrer la fenêtre
        this.setDefaultCloseOperation (JFrame.DISPOSE_ON_CLOSE) ;
        this.setSize (largeur, hauteur) ;
        Fenetres.centrer                         (this, parent) ;
        Fenetres.ajouterEcouteurEchapPourMasquer (this) ;
	}

    /**
     * Une fenêtre pour la bibliographie
     */
	public FenPageHTML(String biblio) 
	{
		super("Bibliographie");
    	//la largeur et la hauteur
    	int largeur = 1200, hauteur = 500;
        // Construire le navigateur
        this.navigHTML = new JEditorPane() ;
        this.navigHTML.setContentType("text/html");
        this.navigHTML.setText(biblio);
        this.navigHTML.setEditable (false) ;
        this.navigHTML.addMouseListener (this) ;
        // Assembler les composants
        this.getContentPane().setLayout (new BorderLayout()) ;
        JScrollPane jsp = new JScrollPane(this.navigHTML);
        this.getContentPane().add (jsp) ;
        // Paramétrer la fenêtre
        this.setDefaultCloseOperation (JFrame.DISPOSE_ON_CLOSE) ;
        this.setSize (largeur, hauteur) ;
        Fenetres.centrer                         (this) ;
        Fenetres.ajouterEcouteurEchapPourMasquer (this) ;
	}


	/**
     * Gestion des clics de souris.
     */
    public void mouseClicked  (MouseEvent evt)
    {
        if (evt.getSource() == this.navigHTML)
            gererClicsHtml (evt.getPoint()) ;
        else
            throw new IllegalStateException ("Source inconnue pour l'événement : " + evt.getSource()) ;
    }
    
    /**
     * Traitement des clics sur les liens.
     * @param posClic Position à laquelle le clic a été effectué.
     */
    public void gererClicsHtml (Point posClic)
    {
        final String PREFIXE_PAGE_COURANTE  = "#" ;      // Début des liens vers un autre Paragraphe

        // Regarder si on a cliqué sur un lien
        HTMLDocument document     = (HTMLDocument) this.navigHTML.getDocument() ;
        int          posDoc       = this.navigHTML.viewToModel (posClic) ;
        Element      eltCaractere = document.getCharacterElement (posDoc) ;
        AttributeSet attrs        = eltCaractere.getAttributes() ;
        AttributeSet lien         = (AttributeSet) attrs.getAttribute (HTML.Tag.A) ;

        // Si on a cliqué sur un lien
        if (lien != null)
        {
            String cibleLien = (String) lien.getAttribute (HTML.Attribute.HREF) ;

            // Lien vers une autre ancre de la page
            if (cibleLien.startsWith (PREFIXE_PAGE_COURANTE))
            {
                try
                {
                    this.navigHTML.setPage (this.urlBasePage + cibleLien) ;
                }
                catch (IOException e)
                {
                    throw new RuntimeException ("Erreur de lecture de l'URL", e) ;
                }
            
            }
            // Lien vers une page externe
            else
            {
                String chemDocumentation = HistoireMondiale.instance().config().chemRepDocumentation() ;
                String urlPage = urlLocaleDeNomFichier (chemDocumentation + File.separator) + "/" + cibleLien ;
                try
                {
                    new FenPageHTML ("", urlPage, 600, 400).setVisible (true) ;
                }
                catch (IOException e)
                {
                    e.printStackTrace() ;
                    System.err.println (e.getMessage()) ;
                    JOptionPane.showMessageDialog (null,
                                                   "Impossible d'accéder au fichier désigné par ce lien\n" +
                                                   "Fichier : " + urlPage + "\n" +
                                                   "Message d'erreur : " + Exceptions.extraireMessagesErreur (e),
                                                   "Erreur de suivi du lien",
                                                   JOptionPane.ERROR_MESSAGE) ;
                }
            }
        }
    }


    
    // Méthodes non implémentées de MouseListener
    public void mouseEntered  (MouseEvent evt) {} ;
    public void mouseExited   (MouseEvent evt) {} ;
    public void mousePressed  (MouseEvent evt) {} ;
    public void mouseReleased (MouseEvent evt) {} ;
}
