/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.igraphique;

import static fr.histoiremondiale.histoire.igraphique.DialCivilisations.OBJET_PARAGRAPHE_BIDON;
import static fr.histoiremondiale.histoire.igraphique.DialCivilisations.TEXTE_AUCUN_PARAGRAPHE;

import java.awt.Color ;
import java.awt.Component ;
import java.awt.GridLayout ;
import java.awt.event.MouseEvent ;
import java.awt.event.MouseListener ;
import java.util.Collections ;
import java.util.List ;
import java.util.Vector ;

import javax.swing.JDialog ;
import javax.swing.JFrame ;
import javax.swing.JLabel ;
import javax.swing.JList ;
import javax.swing.JScrollPane ;
import javax.swing.ListCellRenderer ;
import javax.swing.ListSelectionModel ;
import javax.swing.event.ListSelectionEvent ;
import javax.swing.event.ListSelectionListener ;

import fr.histoiremondiale.histoire.EtatAppli;
import fr.histoiremondiale.histoire.HistoireMondiale;
import fr.histoiremondiale.histoire.donnees.Civilisation;
import fr.histoiremondiale.histoire.igraphique.actions.ActionsCarte;
import fr.histoiremondiale.histoire.igraphique.actions.ActionsNavigateur;
import fr.histoiremondiale.histoire.igraphique.actions.ActionsOptionsGraphiques;
import fr.histoiremondiale.histoire.igraphique.donnees.InfosParagrapheNavig;
import fr.histoiremondiale.histoire.igraphique.donnees.comparateurs.ComparateurParagraphesParDate;
import fr.histoiremondiale.histoire.utiles.igraphique.swing.Fenetres;



/**
 * Boîte de dialogue affichant les civilisation et les paragraphes associés.
 */
public class DialCivilisations extends JDialog implements MouseListener, ListSelectionListener
{

    static final String TEXTE_AUCUN_PARAGRAPHE = "(aucun paragraphe pour cette civilisation)" ;
    static final Object OBJET_PARAGRAPHE_BIDON = null ;     // "Objet" représentant un paragraphe bidon (juste pour mettre vaguement quelque chose dans la liste des paragraphes)
    
    private JList<Object> liste_civilisation ;
    private JList<Object> liste_paragraphes ;


    
    /**
     * Constructeur.
     * @param parent Fenêtre mère.
     */
    public DialCivilisations (JFrame parent)
    {
        super (parent, "Civilisations", false) ;
        
        // Construire la boîte de dialogue
        // (civilisations)
        List<Civilisation> civilisations = HistoireMondiale.instance().donneesHisto().civilisationsTrieesParNom() ;
        this.liste_civilisation = new JList<Object> (new Vector<Civilisation> (civilisations)) ;
        this.liste_civilisation.setCellRenderer          (new GenRenduCivilisation()) ;
        this.liste_civilisation.setSelectionMode         (ListSelectionModel.SINGLE_SELECTION) ;
        this.liste_civilisation.addListSelectionListener (this) ;
        this.liste_civilisation.addMouseListener         (this) ;
        // (paragraphes)
        this.liste_paragraphes = new JList<Object>() ;
        this.liste_paragraphes.setCellRenderer          (new GenRenduParagraphe()) ;
        this.liste_paragraphes.setSelectionMode         (ListSelectionModel.SINGLE_SELECTION) ;
        this.liste_paragraphes.addListSelectionListener (this) ;
        this.liste_paragraphes.addMouseListener         (this) ;
        remplirListeParagraphes (null) ;
        // (assemblage)
        this.getContentPane().setLayout (new GridLayout (1, 2)) ;
        this.getContentPane().add (new JScrollPane (this.liste_civilisation)) ;
        this.getContentPane().add (new JScrollPane (this.liste_paragraphes)) ;
        
        // Paramétrer la boîte de dialogue
        this.setDefaultCloseOperation (JDialog.HIDE_ON_CLOSE) ;
        this.setSize (500, 500) ;
        Fenetres.centrer                         (this, parent) ;
        Fenetres.ajouterEcouteurEchapPourMasquer (this) ;
    }
    
    
    // Remplit la liste des paragraphes avec les paragraphes de la civilisation donnée
    private void remplirListeParagraphes (Civilisation civilisation)
    {
        Vector<Object> modeleListe = null ;
        
        // Construire le modèle de la liste si on a quelque chose à mettre dedans
        if (civilisation != null)
        {
            List<InfosParagrapheNavig> paragraphes = HistoireMondiale.instance().donneesIGraphique().paragraphesParCriteres (civilisation, null) ;
            if (! paragraphes.isEmpty())
            {
                Collections.sort (paragraphes, new ComparateurParagraphesParDate()) ;
                modeleListe = new Vector<Object> (paragraphes) ;
            }
        }
        
        // Si on n'a pas pu remplir le modèle
        if (modeleListe == null)
        {
            modeleListe = new Vector<Object>() ;
            modeleListe.add (OBJET_PARAGRAPHE_BIDON) ;
        }
        
        // Modifier les données affichées
        this.liste_paragraphes.setListData (modeleListe) ;
    }
    
    

    /**
     * Changement dans la sélection des listes.
     */
    public void valueChanged (ListSelectionEvent evt)
    {
        Object source = evt.getSource() ;

        // Sélection d'une nouvelle civilisation
        if (source == this.liste_civilisation)
        {
            // (si le changement est bien terminé)
            if (! evt.getValueIsAdjusting())
            {
                // Afficher les paragraphes associés à la civilisation choisie
                Civilisation civilisationSelectionnee = (Civilisation) this.liste_civilisation.getSelectedValue() ;
                remplirListeParagraphes (civilisationSelectionnee) ;
                // Sélectionner cette civilisation
                ActionsOptionsGraphiques.modifierCivilisationSelectionnee (civilisationSelectionnee) ;
            }
        }
        // Sélection d'un nouveau paragraphe
        else if (source == this.liste_paragraphes)
        {
            // (si le changement est bien terminé)
            if (! evt.getValueIsAdjusting())
            {
                InfosParagrapheNavig paragraphe = (InfosParagrapheNavig) this.liste_paragraphes.getSelectedValue() ;
                if (paragraphe == OBJET_PARAGRAPHE_BIDON)
                    this.liste_paragraphes.clearSelection() ;
                else
                    ActionsNavigateur.affVueCorrespondante (paragraphe) ;
            }
        }
        else
        {
            throw new IllegalStateException ("Source inconnue") ;
        }
        
    }
    

    /**
     * Clics dans la boîte de dialogue.
     */
    public void mouseClicked (MouseEvent evt)
    {
        EtatAppli etatAppli = HistoireMondiale.instance().etat() ;
        Object    source    = evt.getSource() ;
        
        // Double clic sur une civilisation
        if (source == this.liste_civilisation)
        {
            if (evt.getClickCount() == 2)
            {
                int          indiceClique        = this.liste_civilisation.locationToIndex (evt.getPoint()) ;
                Civilisation civilisationCliquee = (Civilisation) this.liste_civilisation.getModel().getElementAt (indiceClique) ;
                ActionsCarte.centrerCarteSurCivilisation (civilisationCliquee, etatAppli.annee()) ;
                ActionsOptionsGraphiques.modifierCivilisationSelectionnee (civilisationCliquee) ;
            }
        }
        // Simple/double sur un paragraphe
        else if (source == this.liste_paragraphes)
        {
            if (  etatAppli.simpleClicFermerParagraphes() && evt.getClickCount() == 1 ||
                ! etatAppli.simpleClicFermerParagraphes() && evt.getClickCount() == 2)
            {
                // Fermer la boîte de dialogue
                this.setVisible (false) ;
            }
        }
        else
        {
            throw new IllegalStateException ("Source inconnue") ;
        }
        
    }
    
    
    // Méthodes non implémentées de MouseListener
    public void mouseEntered  (MouseEvent evt) {}
    public void mouseExited   (MouseEvent evt) {}
    public void mousePressed  (MouseEvent evt) {}
    public void mouseReleased (MouseEvent evt) {}

}


/**
 * Classe permettant d'afficher les éléments de la liste des civilisations.
 */
class GenRenduCivilisation extends JLabel implements ListCellRenderer<Object>
{

    // Renvoie le composant représentant une civilisation
    public Component getListCellRendererComponent (JList<?> lists, Object valeurModele, int indice, boolean estSelectionne, boolean aLeFocus)
    {
        // Récupérer la civilisation concernée
        Civilisation civilisation = (Civilisation) valeurModele ;
        
        // Paramétrer le rendu
        List<InfosParagrapheNavig> paragraphesCivilisation = HistoireMondiale.instance().donneesIGraphique().paragraphesParCriteres (civilisation, null) ;
        String texte   = civilisation.nom() ;
        Color  couleur = (estSelectionne              ? Color.RED :
                         (paragraphesCivilisation.isEmpty() ? Color.GRAY :
                                                        Color.BLACK)) ;
        
        // Renvoyer le composant
        JLabel txt_civilisation = new JLabel (texte) ;
        txt_civilisation.setForeground (couleur) ;
        return txt_civilisation ;
    }
    
}


/**
 * Classe permettant d'afficher les éléments de la liste des paragraphes.
 */
class GenRenduParagraphe extends JLabel implements ListCellRenderer<Object>
{

    // Renvoie le composant représentant un paragraphe
    public Component getListCellRendererComponent (JList<?> lists, Object valeurModele, int indice, boolean estSelectionne, boolean aLeFocus)
    {
        // Récupérer le paragraphe concerné
        InfosParagrapheNavig paragraphe = (InfosParagrapheNavig) valeurModele ;
        
        // Préparer le rendu (texte et couleur)
        String texte ;
        Color  couleur ;
        if (paragraphe == OBJET_PARAGRAPHE_BIDON)
        {
            texte   = TEXTE_AUCUN_PARAGRAPHE ;
            couleur = Color.GRAY ;
        }
        else
        {
            String titre      = paragraphe.titre() ;
            int    debPeriode = paragraphe.anneeDeb() ;
            int    finPeriode = paragraphe.anneeFin() ;
//            texte   = "[" + debPeriode + " -> " + finPeriode + "] "+ titre ;
            texte   = "[" + debPeriode + " à " + finPeriode + "] "+ titre ;
            couleur = (estSelectionne ? Color.RED :
                                        Color.BLACK) ;
        }
        
        // Renvoyer le composant
        JLabel txt_paragraphe = new JLabel (texte) ;
        txt_paragraphe.setForeground (couleur) ;
        return txt_paragraphe ;
    }
    
}
