/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire.donnees;

import java.awt.Color;
import java.util.Set;

import fr.histoiremondiale.histoire.utiles.math.PointSphere;




/**
 * Informations sur un territoire.
 * Il s'agit d'une civilisation dans un contexte donné (une période donnée), il peut y avoir plusieurs
 *   territoires pour la même civilisation, à différentes époques.
 */
public class Territoire
{

    private int                  id ;                // Identifiant
    private int                  rang ;              // Numéro d'ordre du territoire (pour l'ordre de tracé, des territoires du fond aux territoires du dessus de la carte)
    private String               nom ;               // Nom du territoire
    private Set<TexteTerritoire> textes ;            // Textes à afficher sur le territoire, à différentes époques
    private Civilisation         civilisation ;      // Civilisation à laquelle est rattaché le territoire (éventuellement aucune ; certains territoires peuvent ne pas réellement correspondre à une civilisation [c'était le cas des Anciens Rivages pendant un temps par exemple])
    private Pourtour             pourtour ;          // Pourtour du territoire
    private Color                coulInterieur ;     // Couleur de l'intérieur (remplissage de la forme du territoire)
    private Color                coulFrontieres ;    // Couleur des frontières
    private int                  styleHachures ;     // Style des hachures
    private int                  styleFrontieres ;   // Style des frontières
    private int                  dateDebut ;         // Dates de début et de fin d'existence de ce territoire
    private int                  dateFin ;
    private double               valLoupeLimiteAff ; // Valeur limite d'agrandissement pour l'affichage des infos de ce territoire (en dessous de ce niveau d'agrandissement, les informations ne sont pas affichées)
    
    
    
    /**
     * Constructeur.
     * @param id                Identifiant du territoire.
     * @param rang              Rang d'affichage du territoire.
     * @param nom               Nom du territoire.
     * @param textes            Textes à afficher sur le territoire.
     * @param civilisation      Civilisation à laquelle est rattachée le territoire.
     * @param coulInterieur     Couleur de remplissage du territoire (pour le dessin).
     * @param coulFrontieres    Couleur des frontières (pour le dessin).
     * @param styleHachures     Style des hachures (pour le dessin).
     * @param styleFrontieres   Style des frontières (pour le dessin).
     * @param dateDebut         Début de la période pendant laquelle le territoire est affiché.
     * @param dateFin           Fin   de la période pendant laquelle le territoire est affiché.
     * @param valLoupeLimiteAff Valeur minimale de la loupe pour afficher ce territoire.
     * @param pourtour          Pourtour du territoire.
     */
    public Territoire (int id, int rang, String nom, Set<TexteTerritoire> textes,
                       Civilisation civilisation,
                       Color coulInterieur, Color coulFrontieres, int styleHachures, int styleFrontieres,
                       int dateDebut, int dateFin, double valLoupeLimiteAff, Pourtour pourtour)
    {
        this.id                = id ;
        this.rang              = rang ;
        this.nom               = nom ;
        this.textes            = textes ;
        this.civilisation       = civilisation ;
        this.coulInterieur     = coulInterieur ;
        this.coulFrontieres    = coulFrontieres ;
        this.styleHachures     = styleHachures ;
        this.styleFrontieres   = styleFrontieres ;
        this.dateDebut         = dateDebut ;
        this.dateFin           = dateFin ;
        this.valLoupeLimiteAff = valLoupeLimiteAff ;
        this.pourtour          = pourtour ;
    }
    
    /**
     * Constructeur d'un territoire à partir de son id. En général un territoire bidon pour stocker l'id.
     * @param id L'identifiant du territoire.
     */
    public Territoire (int id)
    {
        this.id = id ;
    }
    

    
    /**
     * Renvoie le texte associé au territoire pour une année donnée.<br>
     * Renvoie null s'il n'y a pas de texte pour une année donnée.
     * @param  annee L'année pour laquelle renvoyer le texte.
     * @return Le texte correspondant.
     */
    public TexteTerritoire texteEn (int annee)
    {
        for (TexteTerritoire texte : this.textes)
        {
            if (texte.afficheEn (annee))
                return texte ;
        }
        return null ;
    }
    
    
    
    // Egalité
    // Deux territoires sont égaux uniquement s'ils ont le même id
    public boolean equals (Object o)
    {
        if (o == null)
            return false ;
        if (o == this)
            return true ;
        if (o.getClass() != this.getClass())
            return false ;
        
        return this.id == ((Territoire)o).id ;
    }
    
    
    // Code de hachage
    public int hashCode ()
    {
        return this.id ;
    }
    
    
    
    // Représentation sous forme de chaîne
    public String toString ()
    {
        return "" + this.id + " : " + this.nom + " : " + this.dateDebut + " -> " + this.dateFin ;
    }
    
    //la même chose mais avec le nomp normalisé
    public String enChaine()
    {
    	return "" + this.id + " : " + this.nomNormalise() + " : " + this.dateDebut + " -> " + this.dateFin ;
    }
    
    // Autres accesseurs
    public int                  id                () { return this.id ; }
    public int                  rang              () { return this.rang ; }
    public String               nom               () { return this.nom ; }
    public String               nomNormalise      () { return this.nom.replace ('$', ' ') ; }
    public Set<TexteTerritoire> textes            () { return this.textes ; }
    public Pourtour             pourtour          () { return this.pourtour ; }
    public Color                coulInterieur     () { return this.coulInterieur ; }
    public Color                coulFrontieres    () { return this.coulFrontieres ; }
    public int                  styleHachures     () { return this.styleHachures ; }
    public int                  styleFrontieres   () { return this.styleFrontieres ; }
    public int                  dateDebut         () { return this.dateDebut ; }
    public int                  dateFin           () { return this.dateFin ; }
    public PointSphere          pointCentral      () { return this.pourtour.centreCercleLimite() ; }
    public Civilisation         civilisation      () { return this.civilisation ; }
    public double               valLoupeLimiteAff () { return this.valLoupeLimiteAff ; }
    
}
