/**
 * 
 * Copyright 2010-2020 Patrice Henrio, Sylvain Lavalley
 * 
 * Ce fichier fait partie du logiciel Histoire.
 *
 * Histoire est un logiciel libre : vous pouvez le redistribuer et/ou
 * le modifier sous les termes de la licence Affero GPL publiée par
 * la Fondation pour le logiciel libre (Free Software Foundation), en
 * choisissant la version 3 de cette licence ou n'importe quelle version
 * ultérieure, à votre convenance.
 *
 * Histoire est distribué en espérant qu'il sera utile, mais SANS GARANTIE
 * D'AUCUNE SORTE : y compris d'être vendable ou de pouvoir servir un
 * but donné. Voir le texte de la licence AGPL pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la licence AGPL avec Histoire.
 * Si ce n'est pas le cas, regardez à cette adresse :
 * <http://www.gnu.org/licenses/>.
 *  
 */
package fr.histoiremondiale.histoire;

import java.util.ArrayList ;
import java.util.HashMap ;
import java.util.List ;
import java.util.Map ;

import fr.histoiremondiale.histoire.donnees.Civilisation;
import fr.histoiremondiale.histoire.donnees.Limes;
import fr.histoiremondiale.histoire.donnees.Territoire;
import fr.histoiremondiale.histoire.stockage.AccesDonneesHistoriques;



/**
 * Rassemble les données historiques de l'application.
 */
public class DonneesHistoriques
{
    
    // Données utilisées pour construire le reste
    private Map<Integer,Limes> indexLimes ;     // Limes utilisés pour définir des contours et autres tracés

    
    // Données principales
    private List<Civilisation> civilisations ;                          // Civilisations, triées par nom
    private List<Territoire>   territoires ;                            // Territoires, triés par rang
    
    // Index sur les données principales
    private Map<Integer,Civilisation>   indexCivilisationsParId ;       // Civilisations indexées sur leur id
    private Map<String,Civilisation>    indexCivilisationsParNom ;      // Civilisations indexées sur leur nom
    private Map<Integer,Territoire>     indexTerritoiresParId ;         // Territoires indexés sur leur id

    
    // Accès aux données
    private AccesDonneesHistoriques accesDonneesHistoriques ;
    
    
    
    /**
     * Constructeur.
     * @param accesDonneesHistoriques Objet d'accès aux données historiques de l'application.
     * @param indexLimes              Table d'association id -> limes.
     */
    public DonneesHistoriques (AccesDonneesHistoriques accesDonneesHistoriques, Map<Integer,Limes> indexLimes)
    {
        // Début du chargement des données
        long dateDeb = System.currentTimeMillis() ;
        
        // Initialiser les objets d'accès aux données
        this.accesDonneesHistoriques = accesDonneesHistoriques ;
        
        // Mémoriser les limes
        this.indexLimes = indexLimes ;
        
        // Charger et indexer les civilisations
        this.civilisations = this.accesDonneesHistoriques.chargerCivilisations() ;
        // (civilisations par id)
        this.indexCivilisationsParId = new HashMap<>() ;
        for (Civilisation civilisation : this.civilisations)
            this.indexCivilisationsParId.put (civilisation.id(), civilisation) ;
        // (civilisations par nom)
        this.indexCivilisationsParNom = new HashMap<>() ;
        for (Civilisation civilisation : this.civilisations)
            this.indexCivilisationsParNom.put (civilisation.nom(), civilisation) ;

        // Charger et indexer les territoires
        this.territoires = this.accesDonneesHistoriques.chargerTerritoires (this.indexCivilisationsParId, this.indexLimes) ;
        // (territoires par id)
        this.indexTerritoiresParId = new HashMap<>() ;
        for (Territoire territoire : this.territoires)
            this.indexTerritoiresParId.put (territoire.id(), territoire) ;
        
        // Fin du chargement des données
        long dateFin = System.currentTimeMillis() ;
        
        // Afficher le compte-rendu
        // (à conserver ; si l'utilisateur ne veut rien voir, il n'affiche pas la console)
        System.out.println ("\t\tDonnées historiques chargées en " + (dateFin - dateDeb)  + " ms.") ;
        System.out.println ("\t\t\tNombre de civilisations chargées : " + this.civilisations.size()) ;
        System.out.println ("\t\t\tNombre de territoires chargés    : " + this.territoires.size()) ;
    }
    
    
    /**
     * Renvoie la civilisation d'identifiant donné.<br>
     * Renvoie null si aucune civilisation ne correspond à cet identifiant.
     * @param id L'identifiant de la civilisation recherchée.
     * @return La civilisation correspondante, null si aucune.
     */
    public Civilisation civilisationDId (int id)
    {
        return this.indexCivilisationsParId.get (id) ;
    }
    
    
    /**
     * Renvoie la civilisation de nom donné.<br>
     * Renvoie null si aucune civilisation ne correspond à ce nom.
     * @param nom Le nom de la civilisation recherchée.
     * @return La civilisation correspondante, null si aucune.
     */
    public Civilisation civilisationDeNom (String nom)
    {
        return this.indexCivilisationsParNom.get (nom) ;
    }
    
    
    /**
     * Renvoie la liste des civilisations, triées par nom.
     * @return La liste des civilisations triée.
     */
    public List<Civilisation> civilisationsTrieesParNom ()
    {
        return this.civilisations ;
    }
    
    
    
    /**
     * Renvoie le territoire d'identifiant donné.<br>
     * Renvoie null si aucun territoire ne correspond à cet identifiant.
     * @param id L'identifiant du territoire recherché.
     * @return Le territoire correspondant, null si aucun.
     */
     public Territoire territoireParId (int id)
    {
        return this.indexTerritoiresParId.get (id) ;
    }
    
    
    /**
     * Renvoie la liste des territoires présents à une date donnée.<br>
     * Les territoires sont triés par rang.
     * @param annee L'année pour laquelle faire la recherche.
     * @return La liste des territoires triée.
     */
    public List<Territoire> territoiresAnnee (int annee)
    {
        // Récupérer les territoires concernés
        List<Territoire> territoires = new ArrayList<>() ;
        for (Territoire territoire : this.territoires)
        {
            if (territoire.dateDebut() <= annee && annee <= territoire.dateFin())
                territoires.add (territoire) ;
        }
        
        // Renvoyer le résultat
        return territoires ;
    }
    
    /**
     * accesseur sur la liste des territoires
     * @return la liste des territoires
     */
    public List<Territoire> territoires()
    {
    	return this.territoires;
    }
    
}
